<?php

namespace Inside\ISOD\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Inside\Content\Models\Contents\Users;
use League\Csv\Exception;
use League\Csv\Reader;

class InsideImport extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'inside:isod:import {config} {file} {--silent}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Import from a file into inside';

    /**
     * Importation config
     *
     * @var array
     */
    protected $config;

    /**
     * InsideImport constructor.
     *
     */
    public function __construct()
    {
        parent::__construct();

        $this->config = config('import');
    }

    /**
     *
     */
    public function handle(): void
    {
        /** @var string $configName */
        $configName = $this->argument('config');
        /** @var string $file */
        $file   = $this->argument('file');

        $supported = array_keys($this->config);

        // Check required parameters
        if (! in_array($configName, $supported)) {
            $this->error('config [' . $configName . '] is not supported. Only [' . implode(', ', $supported) . '] are supported type' . (count($supported) > 1 ? 's' : ''));
            return;
        }

        if (! File::exists($file)  || ! File::isFile($file) || ! File::isReadable($file)) {
            $this->error('file [' . $file . '] does not exist or can not be read');
            return;
        }
        $config = $this->config[$configName];

        $type = $config['format']['type'] ?? 'csv';

        if (!in_array($type, ['csv']) || File::extension($file) != $type) {
            $this->error('file type [' . File::extension($file) . '] is not supported');
            return;
        }

        try {
            $csv = Reader::createFromPath($file, 'r');
            $csv->setDelimiter($config['format']['options']['delimiter'] ?? ';');
            $csv->setEnclosure($config['format']['options']['enclosure'] ??'"');
            $csv->setEscape($config['format']['options']['escape'] ?? '\\');
            $csv->setHeaderOffset($config['format']['options']['offset'] ?? null);

            $preFlight = $config['preFlight'];
            if ($preFlight && is_callable($preFlight)) {
                $preFlight($this);
            }

            $records = iterator_to_array($csv->getRecords());
            $this->info(sprintf('Importing %d %s', count($records), str_replace('inside_sodern_', '', $config['table'])));
            $bar = $this->output->createProgressBar(count($records));
            $bar->start();
            $imported = 0;

            foreach ($records as $offset => $record) {
                $datas = [];

                foreach ($config['fields'] as $fieldName => $fieldEquivalent) {
                    if (is_object($fieldEquivalent) && ($fieldEquivalent instanceof \Closure)) {
                        $datas[$fieldName] = $fieldEquivalent($record);
                        if ($datas[$fieldName] === null) {
                            unset($datas[$fieldName]);
                        }
                    } elseif (isset($record[$fieldEquivalent])) {
                        $datas[$fieldName] = $record[$fieldEquivalent];
                    }
                }

                if (array_key_exists('user_uuid', $config['fields']) && (!array_key_exists('user_uuid', $datas) || empty($datas['user_uuid']))) {
                    $datas['user_uuid'] = Users::where('email', 'technique@maecia.com')->firstOrFail()->uuid;
                }

                $bar->advance();

                if (array_key_exists('mandatory', $config) && (!array_key_exists($config['mandatory'], $datas) || empty($datas[$config['mandatory']]))) {
                    continue;
                }

                DB::table($config['table'])->insert($datas);

                $imported++;
            }

            $bar->finish();
            $this->output->newLine();
            $this->info(sprintf('%s import is finished', ucfirst(str_replace('inside_sodern_', '', $config['table']))));
        } catch (Exception $e) {
            $this->error($e->getMessage());
        }
    }
}
