<?php

namespace Inside\ISOD;

use Exception;
use Illuminate\Support\Facades\Log;
use Inside\Content\Models\Contents\Ideas;
use Inside\ISOD\Listeners\WorkflowEventSubscriber;
use Inside\ISOD\SingleStateMarkingStore;
use ReflectionException;
use Symfony\Component\EventDispatcher\EventDispatcher;
use Symfony\Component\Workflow\DefinitionBuilder;
use Symfony\Component\Workflow\Registry;
use Symfony\Component\Workflow\StateMachine;
use Symfony\Component\Workflow\SupportStrategy\InstanceOfSupportStrategy;
use Symfony\Component\Workflow\Transition;
use Symfony\Component\Workflow\Workflow;

class WorkflowStore
{
    protected Registry $registry;

    protected array $config;

    protected EventDispatcher $dispatcher;

    /**
     *
     * WorkflowStore constructor.
     */
    public function __construct(array $config)
    {
        $this->registry = new Registry();
        $this->config = $config;
        $this->dispatcher = new EventDispatcher();

        // Dispatch events
        $subscriber = new WorkflowEventSubscriber();
        $this->dispatcher->addSubscriber($subscriber);

        // Add workflows from our config
        foreach ($this->config as $name => $workflowData) {
            $this->addFromConfig($name, $workflowData);
        }
    }

    /**
     * Get a workflow from our internal registry
     *
     * @param object $subject
     * @return mixed
     */
    public function get(object $subject)
    {
        return $this->registry->get($subject);
    }

    /**
     *
     * @param Workflow $workflow
     * @param string $class
     */
    public function add(Workflow $workflow, string $class): void
    {
        $this->registry->addWorkflow($workflow, new InstanceOfSupportStrategy($class));
    }

    /**
     * Add workflows from config
     *
     * @param string $name
     * @param array $workflowData
     * @throws ReflectionException
     */
    public function addFromConfig(string $name, array $workflowData): void
    {
        try {
            $builder = new DefinitionBuilder();

            foreach (array_keys($workflowData['places']) as $place) {
                $builder->addPlace((string)$place);
            }
            foreach ($workflowData['transitions'] as $transitionName => $transition) {
                $builder->addTransition(new Transition($transitionName, $transition['from'], $transition['to']));
            }
            $definition = $builder->build();

            // MarkingStore
            $markingStore = new SingleStateMarkingStore();

            // Workflow
            if (isset($workflowData['type']) && $workflowData['type'] === 'state_machine') {
                $className = StateMachine::class;
            } else {
                $className = Workflow::class;
            }
            $workflow = new $className($definition, $markingStore, $this->dispatcher, $name);

            // Attach workflow to our ideas!
            $this->add($workflow, Ideas::class);
        } catch (Exception $e) {
            Log::error(
                '[WorkflowStore::addFromConfig] Failed to add worflow ['.$name.'] => '.$e->getMessage()
            );
            throw $e;
        }
    }
}
