<?php

namespace Inside\ISOD\Services;

use Inside\Content\Models\Contents\Ideas;
use Inside\ISOD\WorkflowStore;
use Inside\Permission\Models\User;

/**
 * Class IdeasBox
 *
 * @package  Inside\ISOD\Services
 * @author   Maecia <technique@maecia.com>
 */
class IdeasBox
{
    /**
     * Ideasbox workflow configuration
     *
     * @var array
     */
    protected $workflow = [
        'ideasbox' => [
            'type'            => 'workflow',
            'initial_marking' => 'idea_in_review',
            'places'          => [
                'idea_in_review'          => [
                    'metadata' => [
                        'color' => '#b1b4c7',
                    ],
                ],
                'simple_work_in_progress' => [
                    'metadata' => [
                        'color' => '#f6c342',
                    ],
                ],
                'work_in_progress'        => [
                    'metadata' => [
                        'color' => '#f6c342',
                    ],
                ],
                'work_done'               => [
                    'metadata' => [
                        'color' => '#46790b',
                    ],
                ],
                'work_in_review'          => [
                    'metadata' => [
                        'color' => '#b1b4c7',
                    ],
                ],
                'work_validated'          => [
                    'metadata' => [
                        'color' => '#87cb69',
                    ],
                ],
                'idea_refused'            => [
                    'metadata' => [
                        'color' => '#da4433',
                    ],
                ],
            ],
            'transitions'     => [
                'validate'        => [
                    'from' => 'idea_in_review',
                    'to'   => 'work_in_progress',
                ],
                'simple_validate' => [
                    'from' => 'idea_in_review',
                    'to'   => 'simple_work_in_progress',
                ],
                'reject'          => [
                    'from' => 'idea_in_review',
                    'to'   => 'idea_refused',
                ],
                'work_done'       => [
                    'from' => 'work_in_progress',
                    'to'   => 'work_done',
                ],
                'to_review_work'  => [
                    'from' => 'work_done',
                    'to'   => 'work_in_review',
                ],
                'work_reviewed'   => [
                    'from' => 'work_in_review',
                    'to'   => 'work_validated',
                ],
                'simple_done'     => [
                    'from' => 'simple_work_in_progress',
                    'to'   => 'work_validated',
                ],
            ],
        ],
    ];

    /**
     * @var WorkflowStore
     */
    protected $workflowStore = null;

    /**
     * IdeasBox constructor.
     */
    public function __construct()
    {
    }

    /**
     * Get workflow Store
     *
     * @return \Inside\ISOD\WorkflowStore
     */
    public function getWorkflow()
    {
        if ($this->workflowStore === null) {
            $this->workflowStore = new WorkflowStore($this->workflow);
        }

        return $this->workflowStore;
    }

    /**
     * @param Ideas $idea
     * @param string          $place
     * @return string|null
     */
    public function getTransition(Ideas $idea, string $place): ?string
    {
        $workflow    = $this->getWorkflow()->get($idea);
        $transitions = $workflow->getEnabledTransitions($idea);
        $transition  = null;
        // front send us wanted new place status, so we need to find corresponding transition
        foreach ($transitions as $tr) {
            if ($tr->getTos()[0] == $place) {
                $transition = $tr->getName();
                break;
            }
        }
        return $transition;
    }

    /**
     * Get Initial status
     *
     * @return string
     */
    public function getInitialStatus(): string
    {
        return $this->workflow['ideasbox']['initial_marking'];
    }

    /**
     * Get status color
     *
     * @param string $status
     * @return string
     */
    public function getStatusColor(string $status): string
    {
        return $this->workflow['ideasbox']['places'][$status]['metadata']['color'] ?? '#000';
    }

    public function canEditAndDelete(User $user, array $data): bool
    {
        if (!isset($data['idea_type']) || ! isset($data['idea_status'])) {
            return false;
        }
        $author       = $data['authors']['uuid'] ?? null;
        $collaborator = $data['idea_collaborator']['uuid'] ?? null;

        return // 1- i have role 'validator' or 'super_administrator'
            $user->hasAnyRole('validator|super_administrator')
            || // 2- idea is simple and i'am the author
            ($data['idea_type'] == 'simple' && $author == $user->uuid)
            || // 3- i'am collaborator of that idea and status is inprogress
            ($collaborator == $user->uuid && $data['idea_status'] == 'work_in_progress')
            || // 4- finally this is a complexe idea and there is no collaborator and status is in progress
            ($data['idea_type'] == 'complexe' && $author == $user->uuid && $collaborator === null
                && $data['idea_status'] == 'work_in_progress');
    }
}
