<?php

use Illuminate\Database\Migrations\Migration;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Section;
use Inside\Content\Models\Sections\CallToAction;
use Inside\Host\Bridge\BridgeContent;
use Inside\Host\Bridge\BridgeContentType;
use Inside\Host\Migrations\CreateContentTypes;

class ReworkSectionLinks extends Migration
{
    use CreateContentTypes;

    /** @var array */
    public $types = [
        'link' => [
            'link' => 'section_link',
        ]
    ];

    /** @var array */
    public $subSectionFields = [
        'link_paragraph'
    ];

    public function up(): void
    {
        $this->bootUp(__FILE__, 'rework_section_links');

        foreach ($this->types as $type => $fields) {
            if (!Schema::hasSectionType($type)) {
                continue;
            }

            foreach ($fields as $old => $new) {
                // Add a new field to save current value
                $linkFieldOptions = Schema::getFieldOptions($type, $old);

                $this->getBridge()->contentTypeCreateOrUpdateField(
                    $type,
                    $new,
                    [
                        'type'                => BridgeContentType::TEXT_FIELD,
                        'name'                => $new,
                        'searchable'          => $linkFieldOptions['searchable'] ?? false,
                        'translatable'        => $linkFieldOptions['translatable'] ?? false,
                        'search_result_field' => $linkFieldOptions['search_result_field'] ?? false,
                        'required'            => $linkFieldOptions['required'] ?? false,
                        'weight'              => $linkFieldOptions['weight'] ?? 0,
                        'title'               => $linkFieldOptions['title'] ?? ['fr' => 'Lien'],
                        'description'         => $linkFieldOptions['description'] ?? ['fr' => ''],
                        'widget'              => [
                            'type'     => BridgeContentType::INSIDE_LINK_WIDGET,
                            'settings' => [],
                        ],
                    ],
                    $linkFieldOptions['weight'] ?? null,
                    'paragraph'
                );

                $contentBridge = new BridgeContent();

                $models = call_user_func(section_type_to_class($type) . '::get');
                $subSectionContents = [];

                foreach ($models as $section) {
                    if ($section->sectionable === null) {
                        continue;
                    }
                    if (Schema::isSectionType(class_to_type($section->sectionable))) {
                        $parent = $section->sectionable;

                        $subSectionContents[$parent->sectionable->uuid] = $parent->sectionable;

                        continue;
                    }
                    $this->getConsole()->writeln('<fg=green>Migrating <fg=cyan>[' . $section->sectionable->title . ']</></>');
                    $sections = $this->copySectionField($section, $old, $new, $type);


                    $contentBridge->contentUpdate(
                        class_to_type($section->sectionable),
                        [
                            'uuid'    => $section->sectionable->uuid,
                            'content' => $sections,
                        ]
                    );
                }

                foreach ($subSectionContents as $uuid => $model) {
                    $this->getConsole()->writeln('<fg=green>Migrating <fg=cyan>[' . $model->title . ']</></>');
                    $sections = $this->copySectionField($model, $old, $new, $type);

                    $contentBridge->contentUpdate(
                        class_to_type($model),
                        [
                            'uuid'    => $uuid,
                            'content' => $sections,
                        ]
                    );
                }

                $this->getBridge()->contentTypeDeleteField('paragraph', $type, $old);
            }
        }
    }

    protected function copySectionField(Section $section, string $from, string $to, string $type, bool $delete = false): array
    {
        /** @var \Inside\Content\Models\Content $content */
        $content = Schema::isSectionType(class_to_type($section)) ? $section->sectionable : $section;
        $sections = [];
        foreach ($content->section_content as $section) {
            $sectionType = class_to_type($section);
            $fields      = Schema::getFieldListing($sectionType);
            $data        = ['bundle' => $sectionType, 'pgID' => $section->uuid];
            foreach ($fields as $fieldName) {
                if (in_array($fieldName, $this->subSectionFields)) {
                    $subFieldName = 'section_' . $fieldName;
                    $subData = [];
                    foreach ($section->{$subFieldName} as $subSection) {
                        $subSectionType = class_to_type($subSection);
                        $subFields      = Schema::getFieldListing($subSectionType);
                        $subData        = ['bundle' => $subSectionType, 'pgID' => $subSection->uuid];

                        foreach ($subFields as $subField) {
                            $subData[$subField] = $subSection->{$subField};
                        }

                        if ($subSectionType == $type) {
                            $subData[$to] = $subData[$from];
                            if ($delete) {
                                unset($data[$from]);
                            }
                        }
                    }

                    $data[$fieldName] = $subData;
                } else {
                    $data[$fieldName] = $section->{$fieldName};
                }
            }
            if ($sectionType == $type) {
                $data[$to] = $data[$from];
                if ($delete) {
                    unset($data[$from]);
                }
            }
            $sections[] = $data;
        }

        return $sections;
    }
}
