<?php

use Illuminate\Database\Migrations\Migration;
use Inside\Content\Models\Contents\Newsletters;
use Inside\Content\Models\Contents\VeilleMarketing;
use Inside\Host\Bridge\BridgeContent;
use Inside\Host\Bridge\BridgeContentType;
use Inside\Host\Migrations\CreateContentTypes;
use Inside\Permission\Models\Role;

class UpgradeNewsletters extends Migration
{
    use CreateContentTypes;

    /** @var array */
    protected $newsletterRolesFieldOption = [
        'name'                => 'newsletter_roles',
        'searchable'          => false,
        'translatable'        => false,
        'default'             => false,
        'search_result_field' => false,
        'required'            => false,
        'weight'              => 8,
        'title'               => [
            'fr' => 'Rôles destinataires',
        ],
        'hidden'         => true,
        'widget'              => [
            'type'     => BridgeContentType::STRING_WIDGET,
            'settings' => [
                'size'        => 60,
                'placeholder' => '',
            ],
            'weight'   => 8,
            'group'    => 'group_advanced_options',
        ],
        'type'                => BridgeContentType::TEXT_FIELD,
        'settings'            => [
            'max_length'     => 255,
            'is_ascii'       => false,
            'case_sensitive' => false,
            'cardinality'    => 1,
        ],
    ];

    public function up(): void
    {
        $output = $this->getConsole();

        $output->writeln("<info>Début de Maj Sodern</info> <module>isod-back</module>");

        $bridge = new BridgeContentType();

        try {
            $output->writeln(
                "<info>Suppression du champ</info> <class>test_email</class><info> du type <module>newsletters</module></info>"
            );
            $bridge->contentTypeDeleteField('node', 'newsletters', 'test_email');
            $output->writeln("<comment>Champ <class>test_email</class> correctement effacé</comment> ");

            $output->writeln(
                "<info>Ajout du champ</info> <class>newsletter_roles</class><info> du type <module>newsletters</module></info>"
            );
            $bridge->contentTypeCreateOrUpdateField(
                'newsletters',
                'newsletter_roles',
                $this->newsletterRolesFieldOption
            );
            $output->writeln("<comment>Champ <class>newsletter_roles</class> correctement créé</comment> ");

            $output->writeln(
                "<info>Suppression du champ</info> <class>test_email</class><info> du type <module>veille_marketing</module></info>"
            );
            $bridge->contentTypeDeleteField('node', 'veille_marketing', 'test_email');
            $output->writeln("<comment>Champ <class>test_email</class> correctement effacé</comment> ");

            $output->writeln(
                "<info>Ajout du champ</info> <class>newsletter_roles</class><info> du type <module>veille_marketing</module></info>"
            );
            $bridge->contentTypeCreateOrUpdateField(
                'veille_marketing',
                'newsletter_roles',
                $this->newsletterRolesFieldOption
            );
            $output->writeln("<comment>Champ <class>newsletter_roles</class> correctement créé</comment> ");

            $contentBridge = new BridgeContent();
            $roles         = Role::where('name', config('newsletters.types.newsletters.default_roles'))->get();
            Newsletters::each(
                function ($newsletter) use ($contentBridge, $output, $roles) {
                    $output->writeln(
                        "<comment>Assignation de la newsletter <class>{$newsletter->title}</class> au rôle {$roles->pluck('name')}</comment> "
                    );

                    $contentBridge->contentUpdate(
                        'newsletters',
                        [
                            'uuid'             => $newsletter->uuid,
                            'type'             => 'node',
                            'bundle'           => 'newsletters',
                            'newsletter_roles' => $roles->pluck('id')->toJson(),
                        ]
                    );
                }
            );

            $roles = Role::where('name', config('newsletters.types.veille_marketing.default_roles'))->get();
            VeilleMarketing::each(
                function ($newsletter) use ($contentBridge, $output, $roles) {
                    $output->writeln(
                        "<comment>Assignation de la veille marketing <class>{$newsletter->title}</class> au rôle {$roles->pluck('name')}</comment> "
                    );

                    $contentBridge->contentUpdate(
                        'veille_marketing',
                        [
                            'uuid'             => $newsletter->uuid,
                            'type'             => 'node',
                            'bundle'           => 'veille_marketing',
                            'newsletter_roles' => $roles->pluck('id')->toJson(),
                        ]
                    );
                }
            );

            $output->writeln(
                "<info>Suppression des anciens types de notication</info>"
            );
            \Inside\Notify\Models\NotificationType::where('model', 'Inside\Content\Models\Contents\Newsletters')->where(
                'event',
                'Inside\Content\Events\ContentCreatedEvent'
            )->delete();
            \Inside\Notify\Models\NotificationType::where('model', 'Inside\Content\Models\Contents\VeilleMarketing')
                                                  ->where('event', 'Inside\Content\Events\ContentCreatedEvent')->delete(
                                                  );
        } catch (\Exception $e) {
            $output->writeln("<comment>Tâche(s) échouée(s) : <error>" . $e->getMessage() . "</error></comment>");
        } finally {
            $output->writeln("<comment>Tâche(s) effectuée(s) avec succès</comment>");
        }
    }

    public function down(): void
    {
    }
}
