<?php

use Illuminate\Database\Migrations\Migration;
use Inside\Authentication\Models\User;
use Inside\Notify\Models\NotificationSubscriber;
use Inside\Notify\Models\NotificationType;

class InstallNotificationTypesForIsod extends Migration
{
    public function up(): void
    {
        // Delete default ugly notification Types
        NotificationType::all()->each->delete(); // @phpstan-ignore-line

        // Let's build ours !

        // Première notification
        // Un utilisateur a répondu à mon commentaire

        NotificationType::create(
            [
                'via'       => 'web',
                'default'   => true,
                'event'     => \Inside\Content\Events\CommentCreatedEvent::class,
                'model'     => \Inside\Content\Models\Contents\Comments::class,
                'type'      => 'global',
                'action'    => 'commentCommented',
                'condition' => 'parent_type:news,events,documents,walls|author:CURRENT_USER',
                'multiple'  => true,
                'language'  => false,
                'profile'   => true,
                'data'      => [
                    'title'       => 'notifications.create.comment.comment.title',
                    'description' => 'notifications.create.comment.comment.description',
                    'icon'        => 'comment',
                    'text'        => 'notifications.create.comment.comment.text',
                    'fields'      => ['title'],
                ],
            ]
        );

        // Un utilisateur a commenté mon article
        NotificationType::create(
            [
                'via'       => 'web',
                'default'   => true,
                'event'     => \Inside\Content\Events\CommentCreatedEvent::class,
                'model'     => \Inside\Content\Models\Contents\Comments::class,
                'type'      => 'global',
                'action'    => 'contentCommented',
                'condition' => 'parent_type:news,events,documents,walls|author:CURRENT_USER',
                'multiple'  => true,
                'language'  => false,
                'profile'   => true,
                'data'      => [
                    'title'       => 'notifications.create.comment.content.title',
                    'description' => 'notifications.create.comment.content.description',
                    'icon'        => 'comment',
                    'text'        => 'notifications.create.comment.content.text',
                    'fields'      => ['title'],
                ],
            ]
        );

        // Un utilisateur a liké le commentaire dont l’utilisateur est l’auteur
        NotificationType::create(
            [
                'via'       => 'web',
                'default'   => true,
                'event'     => \Inside\Reaction\Events\ReactionSubscribedEvent::class,
                'model'     => \Inside\Content\Models\Contents\Comments::class,
                'type'      => 'global',
                'action'    => 'like',
                'condition' => 'author:CURRENT_USER',
                'multiple'  => true,
                'language'  => false,
                'profile'   => true,
                'data'      => [
                    'title'       => 'notifications.reaction.like.comment.title',
                    'description' => 'notifications.reaction.like.comment.description',
                    'icon'        => 'like',
                    'text'        => 'notifications.reaction.like.comment.text',
                    'fields'      => ['title'],
                ],
            ]
        );

        // Un actualité a été publiée à la Une
        NotificationType::create(
            [
                'via'       => 'web',
                'default'   => true,
                'event'     => \Inside\Content\Events\ContentCreatedEvent::class,
                'model'     => \Inside\Content\Models\Contents\News::class,
                'type'      => 'global',
                'action'    => 'create',
                'condition' => 'status:1|is_featured:1',
                'multiple'  => false,
                'language'  => false,
                'profile'   => true,
                'data'      => [
                    'title'       => 'notifications.content.news.featured.title',
                    'description' => 'notifications.content.news.featured.description',
                    'icon'        => 'news',
                    'text'        => 'notifications.content.news.featured.text',
                    'fields'      => ['title'],
                ],
            ]
        );

        // On informe également si une actualité n'était pas à la une mais passe à la Une
        NotificationType::create(
            [
                'via'       => 'web',
                'default'   => true,
                'event'     => \Inside\Content\Events\ContentUpdatingEvent::class,
                'model'     => \Inside\Content\Models\Contents\News::class,
                'type'      => 'global',
                'action'    => 'updating',
                'condition' => '*is_featured:1|status:1',
                'multiple'  => false,
                'language'  => false,
                'profile'   => true,
                'data'      => [
                    'title'       => 'notifications.content.news.featured.title',
                    'description' => 'notifications.content.news.featured.description',
                    'icon'        => 'news',
                    'text'        => 'notifications.content.news.featured.text',
                    'fields'      => ['title'],
                ],
            ]
        );

        // # Les workflows

        // Un contenu a été proposé ( via email )
        NotificationType::firstOrCreate(
            [
                'via'      => 'email',
                'default'  => false,
                'event'    => 'Inside\Workflow\Events\ProposalEdited',
                'model'    => 'Inside\Workflow\Models\Proposal',
                'action'   => 'edit',
                'multiple' => false,
                'language' => false,
                'profile'  => false,
            ],
            [
                'data' => [
                    'title'       => 'notifications.workflow.proposals.edited.title',
                    'description' => 'notifications.workflow.proposals.edited.description',
                    'mail'        => [
                        'subject'    => 'notifications.workflow.proposals.edited.mail.subject',
                        'text'       => 'notifications.workflow.proposals.edited.mail.content',
                        'buttonText' => 'notifications.custom.report.documents.mail.buttonText',
                    ],
                ],
            ]
        );

        // Un contenu a été proposé ( via notif web )
        NotificationType::firstOrCreate(
            [
                'via'      => 'web',
                'default'  => false,
                'event'    => 'Inside\Workflow\Events\ProposalEdited',
                'model'    => 'Inside\Workflow\Models\Proposal',
                'action'   => 'edit',
                'multiple' => false,
                'language' => false,
                'profile'  => false,
            ],
            [
                'data' => [
                    'title'       => 'notifications.workflow.proposals.edited.title',
                    'description' => 'notifications.workflow.proposals.edited.description',
                    'icon'        => 'workflows',
                    'text'        => 'notifications.workflow.proposals.edited.text',
                    'fields'      => [
                        'title',
                    ],
                ],
            ]
        );

        // Un contenu proposé a été accepté ( via email )
        NotificationType::firstOrCreate(
            [
                'via'      => 'email',
                'default'  => false,
                'event'    => 'Inside\Workflow\Events\ProposalReviewed',
                'model'    => 'Inside\Workflow\Models\Proposal',
                'action'   => 'accept',
                'multiple' => false,
                'language' => false,
                'profile'  => false,
            ],
            [
                'data' => [
                    'title'       => 'notifications.workflow.proposals.accepted.title',
                    'description' => 'notifications.workflow.proposals.accepted.description',
                    'mail'        => [
                        'subject'    => 'notifications.workflow.proposals.accepted.mail.subject',
                        'text'       => 'notifications.workflow.proposals.accepted.mail.content',
                        'buttonText' => 'notifications.workflow.proposals.accepted.mail.buttonText',
                    ],
                ],
            ]
        );

        // Un contenu proposé a été accepté ( via notif web )
        NotificationType::firstOrCreate(
            [
                'via'      => 'web',
                'default'  => false,
                'event'    => 'Inside\Workflow\Events\ProposalReviewed',
                'model'    => 'Inside\Workflow\Models\Proposal',
                'action'   => 'accept',
                'multiple' => false,
                'language' => false,
                'profile'  => false,
            ],
            [
                'data' => [
                    'title'       => 'notifications.workflow.proposals.accepted.title',
                    'description' => 'notifications.workflow.proposals.accepted.description',
                    'icon'        => 'workflows',
                    'text'        => 'notifications.workflow.proposals.accepted.text',
                    'fields'      => [
                        'title',
                    ],
                ],
            ]
        );

        // Un contenu proposé a été refusé ( via email )
        NotificationType::firstOrCreate(
            [
                'via'      => 'email',
                'default'  => false,
                'event'    => 'Inside\Workflow\Events\ProposalReviewed',
                'model'    => 'Inside\Workflow\Models\Proposal',
                'action'   => 'decline',
                'multiple' => false,
                'language' => false,
                'profile'  => false,
            ],
            [
                'data' => [
                    'title'       => 'notifications.workflow.proposals.declined.title',
                    'description' => 'notifications.workflow.proposals.declined.description',
                    'mail'        => [
                        'subject'    => 'notifications.workflow.proposals.declined.mail.subject',
                        'text'       => 'notifications.workflow.proposals.declined.mail.content',
                        'buttonText' => 'notifications.workflow.proposals.declined.mail.buttonText',
                    ],
                ],
            ]
        );

        // Un contenu proposé a été refusé ( via notif web )
        NotificationType::firstOrCreate(
            [
                'via'      => 'web',
                'default'  => false,
                'event'    => 'Inside\Workflow\Events\ProposalReviewed',
                'model'    => 'Inside\Workflow\Models\Proposal',
                'action'   => 'decline',
                'multiple' => false,
                'language' => false,
                'profile'  => false,
            ],
            [
                'data' => [
                    'title'       => 'notifications.workflow.proposals.declined.title',
                    'description' => 'notifications.workflow.proposals.declined.description',
                    'icon'        => 'workflows',
                    'text'        => 'notifications.workflow.proposals.declined.text',
                    'fields'      => [
                        'title',
                    ],
                ],
            ]
        );

        // On réinscri tout le monde !
        NotificationType::where('profile', true)->where('via', 'web')->each(function ($type) {
            // Subscribe existant users
            User::all()->each(function ($user) use ($type) {
                NotificationSubscriber::firstOrCreate([
                                                          'user_uuid'            => $user->uuid,
                                                          'notification_type_id' => $type->id,
                                                      ]);
            });
        });
    }

    public function down(): void
    {
    }
}
