<?php

use Adldap\Models\Entry;
use Drupal\Core\File\FileSystemInterface;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Inside\Content\Models\Contents\Company;
use Inside\Content\Models\Contents\Services;
use Inside\Content\Models\Contents\Users;
use Inside\Host\Bridge\BridgeContent;
use Symfony\Component\Finder\Finder;

return [
    'sync_attributes' => [
        // Inside => LDAP
        'mail'                => 'mail',
        'created_at'          => function (Entry $user, BridgeContent $bridge) {
            // Example:                            20021127124git chec227.0Z
            return Carbon::createFromFormat("YmdHis.0Z", $user->getFirstAttribute('whencreated'));
        },
        'name'                => function (Entry $user, BridgeContent $bridge) {
            if (!$user->hasAttribute('samaccountname')
                || (($name = $user->getFirstAttribute('samaccountname')) === '')
            ) {
                return Str::random(64); // Use random name
            }

            return $name;
        },
        'is_maintenance'      => function (Entry $user, BridgeContent $bridge) {
            return !$user->hasAttribute('samaccountname') || ($user->getFirstAttribute('samaccountname') === '')
                || ($user->hasAttribute('samaccountname')
                    && $user->getFirstAttribute('extensionattribute15') == 'hide');
        },
        'firstname'           => function (Entry $user, BridgeContent $bridge) {
            return Str::ucfirst(Str::lower($user->getFirstAttribute('givenname')));
        },
        'lastname'            => function (Entry $user, BridgeContent $bridge) {
            return Str::upper($user->getFirstAttribute('sn'));
        },
        'trigram'             => 'initials',
        'phone'               => 'telephonenumber',
        //
        'office'              => 'physicaldeliveryofficename',
        'samaccountname'      => 'samaccountname',
        'ldap_dn'             => 'distinguishedname',

        //  "CN=DUPOOND Pierre,OU=UserSodern_Rle,OU=2015_Utilisateurs,DC=sodern,DC=net"
        'company'             => function (Entry $user, BridgeContent $bridge) {
            // Example SODERN
            if (!$user->hasAttribute('company')) {
                return [];
            }

            $services = [];

            $userCompanies = $user->getAttribute('company');

            foreach ($userCompanies as $companyName) {
                if (empty($companyName)) {
                    continue;
                }

                $companyName = Str::ucfirst(Str::lower($companyName));

                $uuid    = false;
                $company = Company::where('title', $companyName)->first();

                if ($company) {
                    $uuid = $company->uuid;
                } else {
                    $uuidBridge = $bridge->contentInsert(
                        'company',
                        [
                            'type'   => 'node',
                            'bundle' => 'company',
                            'title'  => $companyName,
                            'uid'    => 1,
                        ]
                    );

                    $company = Company::find($uuidBridge);

                    if ($company) {
                        $uuid = $company->uuid;
                    }
                }

                if ($uuid) {
                    $companies[] = $uuid;
                }
            }

            if (!empty($companies)) {
                return $companies;
            }

            return [];
        },

        // Image
        'image'               => function (Entry $user, \Inside\Host\Bridge\BridgeContent $bridge) {
            // Try to determine if user is external or not
            // Example SODERN
            if (!$user->hasAttribute('company') || !$user->hasAttribute('samaccountname')) {
                return ''; // Company is required ..
            }
            $ldaDN = $user->getFirstAttribute('distinguishedname');
            $insideUser  = Users::where('ldap_dn', $ldaDN)->first();
            if ($insideUser && isset($insideUser->image) && !empty($insideUser->image) && Storage::disk('local')->exists($insideUser->image)) {
                return $insideUser->image;
            }

            $userCompanies = $user->getAttribute('company');

            $isExternal = !in_array('sodern', array_map('strtolower', $userCompanies));

            if (!$isExternal) {
                $path = env('ISOD_PICTURES_PATH', '\\\\betelgeuse\\photos\\');
            } else {
                $path = env('ISOD_PRESTA_PICTURES_PATH', '\\\\betelgeuse\\photos\\Prestataires\\');
            }

            if (!File::exists($path) || !File::isReadable($path)) {
                Log::warning(
                    '(LDAP IMPORT) directory [' . $path
                    . '] does not exist or is not readable, check env ISOD_PICTURES_PATH and ISOD_PRESTA_PICTURES_PATH'
                );

                return '';
            }

            // Try to find photo
            $files = Finder::create()->in($path)->depth(0)->files()->name(
                '/^' . $user->getFirstAttribute('samaccountname') . '\.jpe?g/i'
            );

            if ($files->count() != 1) {
                // No photo stored yet or photo conflict name
                return '';
            }

            // We have exactly one file corresponding to our photo, get it !
            $iterator = $files->getIterator();
            $iterator->rewind();
            $photo = $iterator->current();

            // Read current photo
            $file = $photo->getContents();

            if (empty($file)) {
                return '';
            }

            $directory = 'public://users/';
            $fileName = md5($user->getFirstAttribute('samaccountname')) . '.jpg';

            Drupal::service('file_system')->prepareDirectory($directory, FileSystemInterface::CREATE_DIRECTORY);
            Drupal::service('file_system')->saveData($file, $directory . $fileName, FileSystemInterface::EXISTS_REPLACE);

            return 'users/'.$fileName;
        },

        // Service
        'services'            => function (Entry $user, \Inside\Host\Bridge\BridgeContent $bridge) {
            $ldapServiceAttribute = env('ISOD_LDAP_SERVICES_ATTRIBUTE', 'department');

            if (!$user->hasAttribute($ldapServiceAttribute)) {
                return [];
            }

            $services = [];

            $userServices = $user->getAttribute($ldapServiceAttribute);

            foreach ($userServices as $serviceName) {
                if (empty($serviceName)) {
                    continue;
                }

                $uuid    = false;
                $service = Services::where('title', $serviceName)->first();

                if ($service) {
                    $uuid = $service->uuid;
                } else {
                    $uuidBridge = $bridge->contentInsert(
                        'services',
                        [
                            'type'   => 'node',
                            'bundle' => 'services',
                            'title'  => $serviceName,
                            'uid'    => 1,
                        ]
                    );

                    $service = Services::find($uuidBridge);

                    if ($service) {
                        $uuid = $service->uuid;
                    }
                }

                if ($uuid) {
                    $services[] = $uuid;
                }
            }

            if (!empty($services)) {
                return $services;
            }

            return [];
        },
        // Manager opérationnel
        'operational_manager' => function (Entry $user, \Inside\Host\Bridge\BridgeContent $bridge) {
            // Only one manager
            $managerDN = $user->getFirstAttribute('manager');

            if (!$managerDN) {
                return [];
            }
            $manager = Users::where('ldap_dn', $managerDN)->first();

            if ($manager) {
                return [$manager->uuid];
            }

            return [];
        },
        // responsable de prestation
        'prestation_manager' => function (Entry $user, \Inside\Host\Bridge\BridgeContent $bridge) {
            // Only one manager
            $managerDN = $user->getFirstAttribute('manager');

            if (!$managerDN) {
                return [];
            }
            $manager = Users::where('ldap_dn', $managerDN)->first();

            if ($manager) {
                return [$manager->uuid];
            }

            return [];
        },

        // assistant
        'assistant'           => function (Entry $user, \Inside\Host\Bridge\BridgeContent $bridge) {
            $assistantDN = $user->getFirstAttribute('assistant');
            // Only one assitant  !

            if (!$assistantDN) {
                return [];
            }
            $assistant = Users::where('ldap_dn', $assistantDN)->first();

            if ($assistant) {
                return [$assistant->uuid];
            }

            return [];
        },
        'cost_center' => 'extensionattribute12',
        'is_external' => function (Entry $user, \Inside\Host\Bridge\BridgeContent $bridge) {
            if (!$user->hasAttribute('company')) {
                return true;
            }
            $userCompanies = $user->getAttribute('company');

            return !in_array('sodern', array_map('strtolower', $userCompanies));
        }
    ],
];
