<?php

namespace Inside\IPRI\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use ReallySimpleJWT\Encode;
use ReallySimpleJWT\Exception\ValidateException;
use ReallySimpleJWT\Jwt;
use ReallySimpleJWT\Parse;
use ReallySimpleJWT\Validate;

/**
 * Inside IPRI middleware.
 *
 * @category Class
 * @package  Inside\IPRI\Http\Middleware\IPRIMiddleware
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class IpriMiddleware
{
    private array $authorized = [
        'api/v1/public/confucius/authorize' => 'GET',
    ];

    public function handle(Request $request, Closure $next): mixed
    {
        if (str_contains(trim(urldecode($request->getUri()), '/'), 'api/v1/public/confucius')) {
            if (!$this->isAuthorized($request)) {
                Log::info('Access unauthorized', ['uri' => $request->getUri()]);
                abort(403, 'Unauthorized');
            }
        }

        return $next($request);
    }


    /**
     * Check if a given URI can bypass authentication validation
     *
     * @param Request $request
     * @return bool
     */
    private function isAuthorized(Request $request)
    {
        $host = filter_input(INPUT_SERVER, 'HTTP_HOST', FILTER_SANITIZE_URL);
        $https = filter_input(INPUT_SERVER, 'HTTPS', FILTER_SANITIZE_URL);

        $base = $https === 'on' ? 'https' : 'http';
        $host = $base.'://'.$host;
        $requestUri = trim(urldecode($request->getUri()), '/');

        if ($host == $requestUri) {
            return true;
        }

        if ($request->header('token')) {
            /** @var string $token */
            $token = $request->header('token');
            $secret = env('CONFUCIUS_SECRET');

            $jwt = new Jwt($token, $secret);

            $parse = new Parse($jwt, new Validate(), new Encode());

            try {
                $parsed = $parse->validate()
                    ->validateExpiration()
                    ->parse();
            } catch (ValidateException $exception) {
                return false;
            }

            $payload = $parsed->getPayload();

            if ($payload['iss'] === env('APP_URL')) {
                $request->merge(['options' => json_encode(['jwt-token' => true])]);
                return true;
            }
        }

        foreach ($this->authorized as $uri => $method) {
            if (strpos($requestUri, $uri) !== false && $method === $request->getMethod()) {
                return true;
            }
        }

        return false;
    }
}
