<?php

namespace Inside\IPRI\Http\Controllers;

use Illuminate\Http\Request;
use Inside\Content\Models\Sections\Image;
use Inside\Content\Services\Queries\ContentQuery;
use Inside\Permission\Facades\Permission;
use Laravel\Lumen\Routing\Controller;
use ReallySimpleJWT\Exception\ValidateException;
use ReallySimpleJWT\Token;

/**
 * Confucius controller.
 *
 * @category Class
 * @package  Inside\IPRI\Http\Controllers
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class ConfuciusController extends Controller
{
    /**
     * The displayed fields.
     */
    protected array $fieldsDisplayed = [
        'title',
        'uuid',
        'slug',
        'created_at',
        'updated_at',
        'start_date',
        'end_date',
        'is_headline',
        'article_availability',
        'image',
        'content',
        'chapo',
        [
            'articles_categories' => [
                'fields' => [
                    'title',
                    'uuid',
                ],
                'filters' => [
                    'sort' => 'title',
                ],
            ],
        ],
    ];

    /**
     * Filterable fields.
     */
    protected array $fieldsFiltered = [
        'start_date' => 'gt',
        'end_date' => 'lt',
        'article_availability' => 'eq',
        'category' => 'eq',
    ];

    /**
     * Authorize public request on confucius.
     */
    public function confuciusAuthorize(Request $request): string
    {
        $expiration = time() + 3600;

        /** @var string|null $secret */
        $secret = $request->header('secret');

        $payload = [
            'iat' => time(),
            'uid' => 1,
            'exp' => $expiration,
            'iss' => env('APP_URL'),
        ];

        return Token::customPayload($payload, $secret ?: '');
    }

    /**
     * List article_blog contents.
     */
    public function confuciusList(Request $request): array
    {
        Permission::disableAllowedScope();
        unset($this->fieldsDisplayed[array_search('content', $this->fieldsDisplayed)]);

        $fieldsDisplayed = json_encode($this->fieldsDisplayed);

        $request->merge(['fields' => $fieldsDisplayed]);

        $filters = [
            'status' => 1,
            'limit' => $request->get('limit') ? $request->get('limit') : 0,
            'sort' => ['is_headline:asc', 'created_at:desc'],
            'paginate' => 1,
        ];

        foreach ($this->fieldsFiltered as $field => $operator) {
            if (!$request->get($field)) {
                continue;
            }

            if ($field === 'category') {
                $categories = $this->confuciusListCategories(new Request());
                $categoriesTitle = array_column($categories['data'], 'title');
                $categoryKey = array_search($request->get($field), $categoriesTitle);
                $categoryUuid = $categoryKey !== false ? $categories['data'][$categoryKey]['uuid'] : 0;

                $filters['articles_categories'] = ['uuid' => $categoryUuid];
                continue;
            }

            $filters[$field.':'.$operator] = $request->get($field);
        }

        $request->merge(['filters' => json_encode($filters)]);

        $service = new ContentQuery();
        $service->requestQuery('article_blog', $request);
        $articles = $service->transformCollection();

        foreach ($articles['data'] as &$article) {
            unset($article['translations']);
        }

        $unusedData = [
            'first_page_url',
            'from',
            'last_page',
            'last_page_url',
            'next_page_url',
            'path',
            'per_page',
            'prev_page_url',
            'to',
        ];

        $articles = array_diff_key($articles, array_flip($unusedData));

        return $articles;
    }

    /**
     * Get specific article_blog.
     */
    public function confuciusGet(Request $request, string $id): array
    {
        Permission::disableAllowedScope();
        $fieldsDisplayed = json_encode($this->fieldsDisplayed);
        $request->merge(['fields' => $fieldsDisplayed]);

        $service = new ContentQuery();
        $service->find('article_blog', $id, $request);
        $articles = $service->transformCollection();

        unset($articles['data'][0]['translations']);

        return $articles;
    }

    /**
     * List all articles_categories.
     */
    public function confuciusListCategories(Request $request): array
    {
        Permission::disableAllowedScope();
        $request->merge(['fields' => json_encode(['title', 'uuid'])]);
        $service = new ContentQuery();
        $service->requestQuery('articles_categories', $request);
        $categories = $service->transformCollection();

        return $categories;
    }

    /**
     * Get specific article_blog image.
     */
    public function confuciusGetImage(Request $request, string $id): ?string
    {
        Permission::disableAllowedScope();
        $request->merge(['fields' => '["image"]']);
        $uuid = substr(strrchr($request->path(), "/") ?: '', 1);
        $image = null;

        $content = Image::query()->find($uuid);

        if ($content) {
            $image = ['image' => env('APP_STORAGE_PATH').'/'.$content->image];
        } else {
            $query = call_user_func('Inside\\Content\\Models\\Contents\\ArticleBlog::withoutGlobalScopes');
            $content = $query->find($id);

            if ($content && !empty($content->image)) {
                $image = ['image' => env('APP_STORAGE_PATH').'/'.$content->image];
            }
        }

        if (!isset($image, $image['image'])) {
            abort(404, 'Not found');
            return null;
        }

        $img = file_get_contents($image['image']) ?: '';

        $fileExtension = strtolower(substr(strrchr($image['image'], ".") ?: '', 1));

        $contentType = match ($fileExtension) {
            "gif" => "image/gif",
            "png" => "image/png",
            "jpeg", "jpg" => "image/jpeg",
            default => '',
        };

        header('Content-type: '.$contentType);

        return $img;
    }
}
