<?php

declare(strict_types=1);

use Illuminate\Database\Migrations\Migration;
use Inside\Content\Models\Contents\Custom;
use Inside\Host\Bridge\BridgeContent;
use Inside\Layout\Models\Layout;

final class AddCustomContentsAndDeleteLayouts extends Migration
{
    private function createCustom(BridgeContent $bridge, array $custom, string $langcode, ?string $uuidHost = null): ?string
    {
        /** @var Custom|null $existing */
        $existing = Custom::query()
            ->where('langcode', $langcode)
            ->where('custom_slug', $custom['custom_slug'])
            ->first()
        ;

        $data = collect($custom)
            ->map(fn ($value) => $value[$langcode] ?? $value)
            ->all()
        ;

        if ($existing instanceof Custom) {
            $data['uuid'] = $existing->uuid;
        }

        if ($uuidHost) {
            $data['uuid_host'] = $uuidHost;
        }

        $data['langcode'] = $langcode;

        return $bridge->contentInsert('custom', $data, ! isset($data['uuid']));
    }

    public function up(): void
    {
        $customs = [
            [
                'custom_slug' => 'calendar',
                'redirection_page' => 'EventsCalendarPage',
                'title' => [
                    'fr' => 'Évènements du groupe',
                    'en' => 'Group events',
                    'de' => 'Veranstaltungen der Gruppe',
                ],
            ],
            [
                'custom_slug' => 'directory',
                'redirection_page' => 'DirectoryPage',
                'title' => [
                    'fr' => 'Annuaire des collaborateurs',
                    'en' => 'Staff directory',
                    'de' => 'Verzeichnis der Mitarbeiter',
                ],
            ],
            [
                'custom_slug' => 'movements',
                'redirection_page' => 'NominationsListingPage',
                'title' => [
                    'fr' => 'Nominations',
                    'en' => 'Nominations',
                    'de' => 'Nominierungen',
                ],
            ],
            [
                'custom_slug' => 'newcomers',
                'redirection_page' => 'NewcomersListingPage',
                'title' => [
                    'fr' => 'Mouvements',
                    'en' => 'Movements',
                    'de' => 'Bewegungen',
                ],
            ],
            [
                'custom_slug' => 'entity',
                'redirection_page' => 'EntityListingPage',
                'title' => [
                    'fr' => 'Le Groupe de A à Z',
                    'en' => 'The Group from A to Z',
                    'de' => 'Die Gruppe von A bis Z',
                ],
            ],
            [
                'custom_slug' => 'ged',
                'redirection_page' => 'GedPage',
                'title' => [
                    'fr' => 'Bibliothèque',
                    'en' => 'Library',
                    'de' => 'Bibliothek',
                ],
                'custom_introduction' => [
                    'fr' => 'Retrouvez tous les documents, photos, fiches pratiques...',
                    'en' => 'Find all the documents, photos, practical sheets...',
                    'de' => 'Hier finden Sie alle Dokumente, Fotos, Praxisblätter...',
                ],
            ],
            [
                'custom_slug' => 'confucius',
                'redirection_page' => 'ConfuciusPage',
                'title' => [
                    'fr' => 'Interface de publication de Confucius',
                    'en' => 'Confucius publishing interface',
                    'de' => "Konfuzius' Publikationsschnittstelle",
                ],
            ],
            [
                'custom_slug' => 'human-resources',
                'redirection_page' => 'GroupsListingPage',
                'title' => [
                    'fr' => 'Groupes',
                    'en' => 'Groups',
                    'de' => 'Gruppen',
                ],
                'custom_introduction' => [
                    'fr' => 'Retrouvez tous vos groupes',
                    'en' => 'Find all your groups',
                    'de' => 'Finden Sie alle Ihre Gruppen',
                ],
            ],
            [
                'custom_slug' => 'activity',
                'redirection_page' => 'ActivitiesPage',
                'title' => [
                    'fr' => 'Activités',
                    'en' => 'Activities',
                    'de' => 'Aktivitäten',
                ],
                'custom_introduction' => [
                    'fr' => 'Retrouvez ci-dessous les dernières mises à jour et contributions',
                    'en' => 'Find below the latest updates and contributions',
                    'de' => 'Nachfolgend finden Sie die neuesten Updates und Beiträge',
                ],
            ],
            [
                'custom_slug' => 'groups',
                'redirection_page' => 'TeamsListingPage',
                'title' => [
                    'fr' => 'Groupes collaboratifs',
                    'en' => 'Collaborative groups',
                    'de' => 'Kollaborative Gruppen',
                ],
                'custom_introduction' => [
                    'fr' => "Ces espaces vous permettent de collaborer par communautés. N'hésitez pas à proposer la création d'un groupe ! ",
                    'en' => 'hese spaces allow you to collaborate by communities. Do not hesitate to suggest the creation of a group!',
                    'de' => 'In diesen Bereichen können Sie nach Gemeinschaften zusammenarbeiten. Zögern Sie nicht, die Gründung einer Gruppe vorzuschlagen!',
                ],
            ],
        ];

        $bridge = new BridgeContent();

        foreach ($customs as $custom) {
            $language = config('app.locale');
            $originalUuid = $this->createCustom($bridge, $custom, $language);

            /** @var Custom|null $original */
            $original = Custom::find($originalUuid);

            foreach (list_languages() as $language) {
                if ($language === config('app.locale')) {
                    continue;
                }

                $this->createCustom($bridge, $custom, $language, $original?->uuid_host);
            }
        }

        // Delete all layouts
        Layout::query()->delete();
    }
}
