<?php


use Illuminate\Database\Migrations\Migration;
use Inside\Notify\Models\NotificationSubscriber;
use Inside\Notify\Models\NotificationType;
use Inside\Notify\Models\NotificationTypeRole;
use Inside\Permission\Models\Role;
use Symfony\Component\Console\Output\ConsoleOutput;

class AddNotificationTypes extends Migration
{
    public function up(): void
    {
        $console = new ConsoleOutput();
        $notificationTypes = collect();

        // Someone commented something
        $notificationTypes[] = NotificationType::firstOrCreate(
            [
                'via' => 'web',
                'default' => true,
                'event' => 'Inside\Content\Events\CommentCreatedEvent',
                'model' => 'Inside\Content\Models\Contents\Comments',
                'action' => 'contentCommented',
                'type' => 'global',
                'condition' => 'parent_type:news,events',
                'multiple' => true,
                'language' => false,
                'profile' => false,
                'role' => true,
            ],
            [
                'data' => [
                    'title' => 'notifications.create.comment.content.title',
                    'description' => 'admin.notifications.create.comment.content.description',
                    'icon' => 'comment',
                    'text' => 'admin.notifications.create.comment.content.<content_type>.text',
                    'fields' => [
                        'title',
                        'content_type',
                    ],
                ],
            ]
        );

        // Someone answer someone else comment
        $notificationTypes[] = NotificationType::firstOrCreate(
            [
                'via' => 'web',
                'default' => true,
                'event' => 'Inside\Content\Events\CommentCreatedEvent',
                'model' => 'Inside\Content\Models\Contents\Comments',
                'action' => 'commentCommented',
                'type' => 'global',
                'condition' => 'author:CURRENT_USER',
                'multiple' => true,
                'language' => false,
                'profile' => false,
                'role' => true,
            ],
            [
                'data' => [
                    'title' => 'admin.notifications.create.comment.comment.title',
                    'description' => 'admin.notifications.create.comment.comment.description',
                    'icon' => 'comment',
                    'text' => 'admin.notifications.create.comment.comment.<content_type>.text',
                    'fields' => [
                        'title',
                        'content_type',
                    ],
                    'url' => 'ROOT_RELATION:news,events,documents',
                ],
            ]
        );

        // Someone liked something
        $notificationTypes[] = NotificationType::firstOrCreate(
            [
                'via' => 'web',
                'default' => true,
                'event' => 'Inside\Reaction\Events\ReactionSubscribedEvent',
                'model' => null,
                'action' => 'like',
                'condition' => null,
                'multiple' => true,
                'language' => false,
                'profile' => false,
                'role' => true,
            ],
            [
                'data' => [
                    'title' => 'admin.notifications.reaction.like.title',
                    'description' => 'admin.notifications.reaction.like.description',
                    'icon' => 'like',
                    'text' => 'admin.notifications.reaction.like.<content_type>.text',
                    'fields' => [
                        'content_type',
                        'title',
                    ],
                ],
            ]
        );

        $role = Role::where('name', 'super_administrator')->first();
        foreach ($notificationTypes as $type) {
            NotificationTypeRole::firstOrCreate(
                [
                    'notification_type_id' => $type->id,
                    'role_id' => $role->id,
                ]
            );
            foreach ($role->users as $user) {
                NotificationSubscriber::firstOrCreate(
                    [
                        'notification_type_id' => $type->id,
                        'user_uuid' => $user->uuid,
                    ]
                );
            }
        }

        $console->writeln('<info>'.$notificationTypes->count().' Notification types created.</info>');
        $console->writeln('<info>1 Role subscribed.</info>');
        $console->writeln('<info>'.$role->users->count().' Users subscribed.</info>');
    }

    public function down(): void
    {
    }
}
