<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Support\Facades\DB;
use Inside\Content\Facades\Schema as InsideSchema;
use Inside\Content\Models\Field;
use Inside\Host\Migrations\CreateContentTypes;

class FixDrupalConfigs extends Migration
{
    use CreateContentTypes;

    protected function fixConfig(string $key, string $lang = 'fr'): void
    {
        $config = DB::table('config')->where('name', $key)->first();
        if (!$config || !isset($config->data)) {
            return;
        }
        $config = unserialize($config->data);

        if (isset($config['langcode']) && $config['langcode'] == $lang) {
            return; // Already set!
        }

        if (array_key_exists('translatable', $config)) {
            $config['translatable'] = true;
        }
        // Need to be fixed ! change lang
        $config['langcode'] = $lang;
        $this->getConsole()->writeln("<info>Changement de langue pour la clé <class>$key</class></info>");
        DB::table('config')->where('name', $key)->update(
            [
                'data' => serialize($config),
            ]
        );
    }

    /**
     * Fix all config for each $types on $cat domain
     *
     * @param string $cat should be something like node or paragraph
     * @param array  $types drupal type name
     */
    protected function fixConfigs(string $cat, array $types): void
    {
        foreach ($types as $type) {
            if (in_array($type, ['comments'])) {
                continue;
            }

            if ($cat == 'paragraph') {
                $this->fixConfig('paragraphs.paragraphs_type.' . $type);
            } elseif ($type == 'users') {
                $this->fixConfig('user.settings');
                $this->fixConfig('user.mail');
                $this->fixConfig('user.flood');
            } else {
                $this->fixConfig($cat . '.type.' . $type);
            }
            $this->fixConfig(
                'language.content_settings.' . ($type == 'users' ? 'user' : $cat) . '.' . ($type == 'users' ? 'user'
                    : $type)
            );

            // Fix overrides
            DB::table('config')->where(
                'name',
                'like',
                'core.base_field_override.' . ($type == 'users' ? 'user' : $cat) . '.' . ($type == 'users' ? 'user'
                    : $type) . '.%'
            )->get()->each(
                function ($config) {
                    $this->fixConfig($config->name);
                }
            );
            // Fix special body config
            $this->fixConfig('field.field.node.' . $type . '.body');

            // Fix each fieldsfor this type
            foreach (InsideSchema::getFieldListing($type) as $field) {
                // Storage
                $this->fixConfig('field.storage.' . ($type == 'users' ? 'user' : $cat) . '.field_' . $field);

                // Field
                $this->fixConfig(
                    'field.field.' . ($type == 'users' ? 'user' : $cat) . '.' . ($type == 'users' ? 'user' : $type)
                    . '.field_' . $field
                );
            }
        }
    }

    public function up(): void
    {
        // Content types
        $this->fixConfigs('node', InsideSchema::getContentTypes());

        // Sections
        $this->fixConfigs('paragraph', InsideSchema::getSectionTypes());

        // Comments
        $this->fixConfig('comment.type.comments');
        foreach (
            ['comment.comment_body', 'comment.field_file', 'node.field_comments', 'node.field_comments_text',] as $storage
        ) {
            $this->fixConfig('field.storage.' . $storage);
        }
        foreach (['field_file', 'comment_body'] as $field) {
            $this->fixConfig('field.field.comment.comments.' . $field);
        }
        $this->fixConfig('language.content_settings.comment.comments');

        // Fix body storage
        $this->fixConfig('field.storage.node.body');

        // Fix inside fields titles & descriptions
        Field::each(
            function ($field) {
                $options = $field->options;
                if (!isset($options['title']['fr'])) {
                    $options['title']['fr'] = $options['title']['en'] ?? '';
                }
                if (!isset($options['description']['fr'])) {
                    if (!is_array($options['description'])) {
                        $options['description'] = ['fr' => ''];
                    }
                    $options['description']['fr'] = $options['description']['en'] ?? '';
                }
                $field->options = $options;
                $field->save();
            }
        );
    }

    public function down(): void
    {
    }
}
