import DefaultTestE2E from '@maecia/inside-tests/e2e/class/DefaultTestE2E'
import e2eConfig from '@maecia/inside-tests/e2e/helpers/e2e-config'
import get from 'lodash.get'

export default class DecommissioningE2E extends DefaultTestE2E {
  constructor(props) {
    super(props)

    this.testName = 'Inside Content'
    this.steps = [
      'before',
      'createContent',
      'checkContent',
      'postComment',
      'checkComment',
      'logoutAdministrator',
      'decommissionedUser',
      'checkContents',
      'checkDecommissionedUserInDirectory',
      'checkDecommisionnedUserEdition',
      'logoutSuperAdministrator',
      'tryUserReconnection',
      'recommissionedUser',
      'retryUserConnection',
      'checkNotification',
      'tryDecommissionningSuperAdmin',
      'checkSuperAdminUserEdition',
      'cleanup'
    ]

    this.config = e2eConfig('inside-user')

    window.localStorage.setItem(`sid2-${Cypress.env('APP_CODE')}--login-component`, 'login-classic-customs')

    this.createdContent = null
    this.createdComment = null
    this.administratorUser = null
    this.superAdministratorUser = null
  }

  before() {
    before(() => {
      if (
        !window.localStorage.getItem(
          `sid2-${Cypress.env('APP_CODE')}--api-token`
        )
      ) {
        cy.start('administrator', true)
      } else {
        cy.setEnvFromStorage()
      }
    })
  }

  createContent() {
    it('Create news content', () => {
      this.administratorUser = JSON.parse(
        window.localStorage.getItem(`sid2-${Cypress.env('APP_CODE')}--current-user`)
      )

      cy.getHttp('/api/v1/content/news_categories', {
        filters: {
          limit: 1
        }
      }).then(categories => {
        cy.factory('news').then(content => {
          content.authors = [Cypress.env('CURRENT_USER_UUID')]
          content.langcode = Cypress.env('CURRENT_LANG')
          content.status = 1
          content.title = 'Une news de decommissionnement'
          content.news_categories = categories.body.data[0].uuid

          cy.postHttp('api/v1/content/news', content).then(res => {
            this.createdContent = res.body
          })
        })
      })
    })
  }

  checkContent() {
    it('Check content', () => {
      cy.intercept(`/api/v1/layout?path=${this.createdContent.slug[0]}`).as('getLayout')
      cy.intercept('/api/v1/content/news').as('getNews')
      cy.intercept('/api/v1/content/comments').as('getComments')

      cy.visit(`/${this.createdContent.slug[0]}`)

      cy.wait(['@getLayout', '@getNews', '@getComments']).spread(() => {
        cy.get('.full-news-container .title').invoke('text').should('eq', this.createdContent.title)
      })
    })
  }

  postComment() {
    it('Post comment', () => {
      cy.postHttp('api/v1/content/comments', {
        body: 'Un commentaire de décommissionnement',
        bundle: 'news',
        fields: ['uuid', 'body'],
        file: null,
        langcode: 'fr',
        news: [this.createdContent.uuid],
        uid: Cypress.env('CURRENT_USER_UUID')
      }).then(res => {
        this.createdComment = res.body
      })
    })
  }

  checkComment() {
    it('Check comment', () => {
      if (this.config.useComments) {
        cy.intercept('/api/v1/content/news').as('getNews')
        cy.intercept('/api/v1/content/comments').as('getComments')

        cy.reload()

        cy.wait(['@getNews', '@getComments']).spread(() => {
          cy.get('.full-comments__feed .answer-comments__text p').invoke('text').should('eq', this.createdComment.body)
        })
      }
    })
  }

  logoutAdministrator() {
    it('Logout administrator user', () => {
      cy.visit('/logout')
    })
  }

  decommissionedUser() {
    it('Decommissionned User', () => {
      cy.start('super_administrator', false)

      cy.putHttp(`/api/v1/content/users/${this.administratorUser.uuid}`, {
        status: 0
      })
    })
  }

  checkContents() {
    it('Check content is already accessible', () => {
      cy.intercept('/api/v1/content/news').as('getNews')
      cy.intercept('/api/v1/content/comments').as('getComments')

      cy.visit(`/${this.createdContent.slug[0]}`)

      cy.wait(['@getNews', '@getComments']).spread(news => {
        assert.isNotNull(news.response.body.data)
      })
    })

    it('Check decommissioned news author', () => {
      if (this.config.useAuthorCard) {
        cy.get('.author .card-users__infos .name').invoke('text').should('contains', `${this.administratorUser.firstname} ${this.administratorUser.lastname}`)
      }
    })

    it('Check decommissioned comment author', () => {
      if (this.config.useComments) {
        cy.get(`.full-comments__feed [data-scroll*=${this.createdComment.uuid}] .answer-comments__author`).invoke('text').should('contains', 'Utilisateur Anonyme')

        /**
         * Post comment to test notification
         */
        cy.postHttp('api/v1/content/comments', {
          body: 'Une réponse de décommissionnement',
          bundle: 'news',
          fields: ['uuid', 'body'],
          file: null,
          langcode: 'fr',
          news: [this.createdContent.uuid],
          uid: Cypress.env('CURRENT_USER_UUID'),
          pid: this.createdComment.uuid
        })
      }
    })
  }

  checkDecommissionedUserInDirectory() {
    it('Check decommissionned user is not in directory', () => {
      cy.intercept(`/api/v1/layout?path=${this.config.directoryPath[Cypress.env('CURRENT_LANG')]}`).as('getLayout')
      cy.intercept('/api/v1/content/users').as('getUsers')

      cy.visit(`/${this.config.directoryPath[Cypress.env('CURRENT_LANG')]}`)

      cy.wait(['@getLayout', '@getUsers'])
      cy.wait(2000)

      const search = `${this.administratorUser.firstname} ${this.administratorUser.lastname}`
      cy.intercept({
        url: 'api/v1/search',
        query: {
          query: search
        }
      }).as('getDecommissionedUser')

      cy.get('.filter-item-container .search input').type(search)

      cy.wait('@getDecommissionedUser').then(res => {
        const users = res.response.body.data.filter(user => user.uuid === this.administratorUser.uuid)

        assert.equal(users.length, 0)
      })
    })
  }

  checkUserFields(currentUser, checkIfNull = true) {
    cy.intercept('/api/v1/form/users').as('getFields')

    cy.visit(`/edit/users/${currentUser.uuid}`)

    cy.wait('@getFields').then(fields => {
      const decommissionedFields = ['lastname', 'firstname', 'name', 'email', 'is_maintenance']

      const flattenObject = function (ob) {
        const toReturn = {}

        for (const i in ob) {
          if (!ob.hasOwnProperty(i)) continue

          if (typeof ob[i].fields === 'object') {
            const flatObject = flattenObject(ob[i].fields)

            for (const x in flatObject) {
              if (!flatObject.hasOwnProperty(x)) continue

              toReturn[`${i}-${x}`] = flatObject[x]
            }
          } else {
            toReturn[i] = ob[i]
          }
        }
        return toReturn
      }

      const flattenData = flattenObject(fields.response.body.data)
      let counter = 0

      Object.values(flattenData).forEach((field, index) => {
        if ((field.value && typeof field.value !== 'object') || (field.value && field.value.length > 0)) {
          counter++

          if (checkIfNull) {
            console.log(field.name, field.value)
            assert.include(decommissionedFields, field.name)
          }
        }
      })

      if (!checkIfNull) {
        assert.isAtLeast(counter, decommissionedFields.length)
      }
    })
  }

  checkDecommisionnedUserEdition() {
    it('Check decommissionned user in edition', () => {
      this.checkUserFields(this.administratorUser)
    })
  }

  logoutSuperAdministrator() {
    it('Logout super administrator user', () => {
      this.superAdministratorUser = JSON.parse(
        window.localStorage.getItem(`sid2-${Cypress.env('APP_CODE')}--current-user`)
      )

      cy.visit('/logout')

      cy.wait(2000)
    })
  }

  connection(email, pwd) {
    cy.intercept('/api/v1/layout?path=login').as('getLayout')

    cy.visit('/login')

    cy.wait('@getLayout')

    // cy.get('.account-users__link').click()

    cy.intercept({
      url: 'api/v1/authentication/login'
    }).as('getLogin')

    cy.get('.input-email input')
      .clear()
      .type(email, { log: false })

    cy.get('.input-password input')
      .clear()
      .type(pwd, { log: false })

    cy.get('#login_form input[type=submit]').click()
  }

  tryUserReconnection() {
    it('Try decommissioned user reconnection', () => {
      this.connection(this.administratorUser.email, this.administratorUser.password)

      cy.wait('@getLogin').then(login => {
        cy.get('.default-toast--error .description').invoke('text').should('contains', 'Votre compte est inactif')
      })
    })
  }

  recommissionedUser() {
    it('Recommissioned user', () => {
      cy.start('super_administrator', false)

      cy.putHttp(`/api/v1/content/users/${this.administratorUser.uuid}`, {
        status: 1
      })

      cy.visit('/logout')
      cy.wait(2000)
    })
  }

  retryUserConnection() {
    it('Try recommissioned user reconnection', () => {
      this.connection(this.administratorUser.email, this.administratorUser.password)

      cy.wait('@getLogin')
      cy.wait(2000)
    })
  }

  checkNotification() {
    it('Check last notification is not triggered during decommissioning', () => {
      cy.intercept('/api/v1/layout?path=home').as('getLayout')
      cy.intercept('/api/v1/notifications').as('getNotifications')

      cy.visit('/')

      cy.wait(['@getLayout', '@getNotifications']).spread((layout, notifications) => {
        const lastNotification = notifications.response.body.data[0]

        if (lastNotification) {
          assert.notEqual(get(lastNotification, 'comment.pid'), this.createdComment.uuid)
        }
      })

      cy.visit('/logout')
      cy.wait(2000)
    })
  }

  tryDecommissionningSuperAdmin() {
    it('Try decommissionned Super Admin', () => {
      cy.start('super_administrator', false)

      cy.putHttp(`/api/v1/content/users/${this.superAdministratorUser.uuid}`, {
        status: 0
      })

      cy.wait(2000)
    })
  }

  checkSuperAdminUserEdition() {
    it('Check Super Admin user fields value', () => {
      this.checkUserFields(this.superAdministratorUser, false)
    })
  }

  cleanup() {
    it('Cleanup', () => {
      cy.cleanup()
    })
  }
}
