<?php

namespace Inside\User\Services;

use Inside\User\Models\User;
use Ramsey\Uuid\Uuid;

class UserService
{
    private const WHITELIST = [
        'uuid',
        'uuid_host',
        'name',
        'email',
        'password',
        'status',
        'langcode',
        'created_at',
        'updated_at',
    ];

    /**
     * Type of the user
     *
     * @var string
     */
    protected $type;

    /**
     * Flattened datas
     *
     * @var array
     */
    protected $data;

    /**
     * @param string $type
     * @param mixed  $extractor
     */
    public function __construct(string $type, $extractor)
    {
        $this->type = $type;
        $this->data = $extractor->getData();
    }

    /**
     * @return User|null
     */
    public function update(): ?User
    {
        $user = $this->load();
        if ($user) {
            $this->process($user, $this->data);

            return $user;
        }

        return $this->create();
    }

    /**
     * @return User|null
     */
    protected function load(): ?User
    {
        if (! isset($this->data['uuid'])) {
            return null;
        }

        return User::find($this->data['uuid']);
    }

    /**
     * @param User $user
     * @param array $data
     * @return void
     */
    protected function process(User $user, array $data)
    {
        $user->update($data);
    }

    public function create(): User
    {
        $user = $this->load();
        $this->data = array_intersect_key($this->data, array_flip(self::WHITELIST));
        if ($user === null) {
            $this->data['uuid'] = Uuid::uuid4();
            $user = User::create($this->data);
        }
        $this->process($user, $this->data);

        return $user;
    }

    /**
     * @return void
     * @throws \Exception
     */
    public function delete(): void
    {
        $user = $this->load();
        if (! $user) {
            return;
        }
        $user->delete();
    }
}
