<?php

namespace Inside\User\Providers;

use Illuminate\Auth\Passwords\PasswordBrokerManager;
use Inside\Authentication\Events\UserDecisionized;
use Inside\Authentication\Events\UserUndecisionized;
use Inside\Content\Events\ContentFullyInsertedEvent;
use Inside\Support\EventServiceProvider;
use Inside\User\Console\CleanUserReferences;
use Inside\User\Console\DisableExpiredUsersCommand;
use Inside\User\Console\SendPasswordCreateMails;
use Inside\User\Listeners\ContentCreatedListener;
use Inside\User\Listeners\DecisionizeUser;
use Inside\User\Listeners\UndecisionizeUser;
use Psr\Container\ContainerExceptionInterface;
use Psr\Container\NotFoundExceptionInterface;

class UserServiceProvider extends EventServiceProvider
{
    protected array $listen = [
        ContentFullyInsertedEvent::class => [
            ContentCreatedListener::class,
        ],
        UserDecisionized::class          => [
            DecisionizeUser::class,
        ],
        UserUndecisionized::class        => [
            UndecisionizeUser::class,
        ],
    ];

    protected array $commands = [
        DisableExpiredUsersCommand::class,
        SendPasswordCreateMails::class,
        CleanUserReferences::class,
    ];

    public function register(): void
    {
        try {
            $this->mergeConfigFrom(__DIR__.'/../../config/users.php', 'users');
            $this->mergeRecursiveConfigFrom(__DIR__.'/../../config/scheduler.php', 'scheduler');
        } catch (NotFoundExceptionInterface|ContainerExceptionInterface) {
        }

        /**
         * Bind our auth broker
         */
        $this->app->singleton(
            'auth.password',
            function ($app) {
                return new PasswordBrokerManager($app);
            }
        );

        $this->app->bind(
            'auth.password.broker',
            function ($app) {
                return $app->make('auth.password')->broker();
            }
        );
    }
}
