<?php

namespace Inside\User\Listeners;

use Exception;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Support\Str;
use Inside\Content\Events\ContentFullyInsertedEvent;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Contents\Users;
use Inside\Host\Bridge\BridgeContent;
use Inside\Notify\Listeners\BaseNotificationListener;
use Inside\User\Jobs\SendCreatePasswordMail;

/**
 * Handle default shortcuts for new users
 */
class ContentCreatedListener extends BaseNotificationListener implements ShouldQueue
{
    public function handle(ContentFullyInsertedEvent $event): void
    {
        $userModel = $event->model;
        if (! $userModel instanceof Users) {
            return;
        }
        $fields = ['shortcuts_menus', 'tools'];

        foreach ($fields as $field) {
            $this->applyDefaultShortcuts($field, $userModel);
        }

        if (Schema::hasField('users', 'send_mail') && $userModel->send_mail) {
            SendCreatePasswordMail::dispatch($userModel->uuid);
        }
    }

    /**
     * @throws Exception
     */
    protected function applyDefaultShortcuts(string $field, Users $user): void
    {
        // Check if the shortcuts_menus field exists on users
        if (! Schema::hasField('users', $field)) {
            return;
        }

        // The user already has shortcuts menus
        if ($user->{Str::camel($field)}->count() > 0) {
            return;
        }

        $options = Schema::getFieldOptions('users', $field);
        $defaultShortcuts = [];

        foreach ($options['default'] as $uuid_host) {
            $shortcut = call_user_func(type_to_class($field).'::query')->where('uuid_host', $uuid_host)->first();
            if (! $shortcut) {
                continue;
            }
            $defaultShortcuts[] = $shortcut->uuid;
        }

        if (empty($defaultShortcuts)) {
            return;
        }
        $bridge = new BridgeContent();
        $bridge->contentUpdate(
            'users',
            [
                'uuid' => $user->uuid,
                $field => $defaultShortcuts,
            ]
        );
    }
}
