<?php

namespace Inside\User\Jobs;

use Carbon\Carbon;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Inside\Authentication\Models\Token;
use Inside\Console\Command;
use Inside\Content\Models\Contents\Users;
use Inside\Host\Bridge\BridgeContent;
use Inside\Jobs\Bus\Dispatchable;
use Inside\Permission\Models\Role;

class DisableExpiredUsersJob implements ShouldQueue
{
    use Dispatchable;
    use InteractsWithQueue;
    use Queueable;
    use SerializesModels;

    /**
     * @var Command|null
     */
    private $command;

    /**
     * @var bool
     */
    protected $simulating;

    /**
     * @param Command|null $command
     */
    public function __construct(Command $command = null, bool $simulating = false)
    {
        $this->command = $command;
        $this->simulating = $simulating;
    }

    /**
     * @param string $text
     * @return void
     */
    protected function info(string $text)
    {
        if ($this->command) {
            $this->command->info($text);
        } else {
            Log::info("[DisableExpiredUsersJob] {$text}");
        }
    }

    /**
     * @return void
     * @throws \Exception
     */
    public function handle()
    {
        $this->info('Début de décommissionement des utilisateurs inactifs');
        if ($this->simulating) {
            $this->command ? $this->command->warning('Mode simulation activé') : Log::info('Mode simulation activé');
        }
        $limitDate = Carbon::now()->modify('-'.config('users.auto_decommissioning_time'));
        if ($limitDate === false) {
            $this->info(
                'Impossible d\'appliquer une modification de data avec ['.config('users.auto_decommissioning_time')
                .']'
            );

            return;
        }
        $rolesWhitelist = config('users.roles_whitelist', []);
        $toDisable = Users::addSelect(
            [
                'has_whitelist_role' => Role::selectRaw('count(*)')->join(
                    'inside_users_roles',
                    'inside_users_roles.role_id',
                    '=',
                    'inside_roles.id'
                )->whereColumn('inside_content_users.uuid', 'inside_users_roles.user_uuid')->whereIn(
                    'name',
                    $rolesWhitelist
                )->take(1),
                'last_login'         => Token::select('created_at')->whereColumn(
                    'user_uuid',
                    'inside_content_users.uuid'
                )->latest()->take(1),
            ]
        )->having('last_login', '<=', $limitDate)->orHavingRaw(
            '`last_login` IS NULL AND `inside_content_users`.`created_at` <= ?',
            [$limitDate]
        )->having('has_whitelist_role', 0)->where('status', 1)->where('is_maintenance', 0)->get();

        $this->info("{$toDisable->count()} users found to disable");

        if ($this->command) {
            $bar = $this->command->getOutput()->createProgressBar($toDisable->count());
            $bar->setFormat("%message%\n %current%/%max% [%bar%] %percent:3s%%");
            $this->command->info('Starting to disable users...');
        }

        $bridge = new BridgeContent();
        foreach ($toDisable as $user) {
            if (! $this->simulating) {
                $bridge->contentUpdate('users', ['uuid' => $user->uuid, 'status' => 0]);
            }
            if ($this->command) {
                $bar->setMessage(" => {$user->name} [{$user->email}]");
                $bar->advance();
            }
            Log::info("[DisableExpiredUsersJob] disabling {$user->name} [{$user->email}]");
        }

        if ($this->command) {
            $this->command->info('');
            $bar->setMessage('DONE');
            $bar->finish();
            $this->command->writeln('');
            $this->command->writeln(str_repeat('-', 80));
        }
        $this->info('Décommissionement des utilisateurs inactifs terminé');
    }
}
