<?php

namespace Inside\Import\Services;

use Illuminate\Database\QueryException;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Inside\Authentication\Models\User;
use Inside\Facades\Http;
use Inside\Import\Exceptions\InvalidReferenceException;
use Inside\Import\Models\UserReference;
use Inside\Support\Str;

/**
 * Class UserImportService
 */
class UserImportService
{
    /**
     * Load user from $data using $entry provider
     * @throws InvalidReferenceException
     */
    public function getUserFromEntry(string $providerType, string $providerName, array $entry, array $data): ?User
    {
        if (! isset($entry['discovering_attribute']) || $entry['discovering_attribute'] == 'email') {
            return User::where('email', $data['email'])->first();
        }
        $reference = $entry['discovering_attribute'];
        if (! isset($data[$reference])) {
            throw InvalidReferenceException::withData($data, $reference);
        }
        Log::channel('import')->info(
            '[UserImportService::getUserFromEntry] Getting user from reference ['.$reference.'] on provider ('
            .$providerType.','.$providerName.') with <'.$data[$reference].'>'
        );

        $userReference =
            UserReference::where('provider_type', $providerType)->where('provider_name', $providerName)->where(
                'reference',
                $data[$reference]
            )->first();
        if ($userReference === null) {
            Log::channel('import')->info(
                '[UserImportService::getUserFromEntry] not found <'.$providerType.'> <'.$providerName.'> <'
                .$data[$reference].'>'
            );

            return null;
        }
        Log::channel('import')->info(
            '[UserImportService::getUserFromEntry] found <'.($userReference->user->uuid ?? '???').'>'
        );

        return $userReference->user;
    }

    /**
     * Do we need to rebuild references
     */
    public function rebuildReferencesIsNeeded(string $providerType, string $providerName): bool
    {
        return UserReference::where('provider_type', $providerType)->where('provider_name', $providerName)->count()
            == 0;
    }

    /**
     * Get reference name from entry
     */
    protected function getReferenceNameFromEntry(array $entry): ?string
    {
        return $entry['discovering_attribute'] ?? null;
    }

    /**
     * set a user reference ( to be used on user creation )
     *
     * IMPORTANT: $entry['discovering_attribute'] is an immuable unique identifier key
     *
     * @throws InvalidReferenceException
     * @throw QueryException
     */
    public function setUserReference(
        string $providerType,
        string $providerName,
        array $entry,
        array $data,
        string $uuid
    ): bool {
        if (! isset($entry['discovering_attribute']) || $entry['discovering_attribute'] == 'email') {
            return true;
        }
        $referenceName = $this->getReferenceNameFromEntry($entry);
        if ($referenceName === null || ! isset($data[$referenceName]) || empty($data[$referenceName])) {
            throw InvalidReferenceException::withData($data, $referenceName);
        }
        $userReference = new UserReference();
        $userReference->provider_type = $providerType;
        $userReference->provider_name = $providerName;
        $userReference->user_id = $uuid;
        $userReference->reference = $data[$referenceName];

        try {
            if ($userReference->save()) {
                return true;
            }
        } catch (QueryException $e) {
            return false;
        }

        return false;
    }

    public function setUserAvatar(?string $url, ?User $user): ?string
    {
        if ($url === null || $user === null) {
            return null;
        }

        $response = Http::get($url);

        if (! $response->successful()) {
            return null;
        }

        $image = $response->body();

        if (md5($image) === md5_file(storage_path('app/'.$user->information->image))) {
            return $user->information->image;
        }

        $mimeType = (new \finfo(FILEINFO_MIME_TYPE))->buffer($image);

        $extension = match ($mimeType) {
            'image/jpeg' => 'jpeg',
            'image/png' => 'png',
            'image/gif' => 'gif',
            'image/webp' => 'webp',
            'image/svg+xml' => 'svg',
            default => null,
        };

        if ($extension === null) {
            return null;
        }

        $path = sprintf('app/chunks/%s', Str::random(32));
        $full = sprintf('%s/avatar-%s.%s', $path, $user->uuid, $extension);

        File::makeDirectory(storage_path($path), 0755, true);
        File::put(storage_path($full), $image);

        return str($full)->after('app/')->toString();
    }
}
