<?php

namespace Inside\Import\Services;

use Exception;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Log;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Content;
use Inside\Database\Eloquent\Builder;
use Inside\Host\Bridge\BridgeContent;

/**
 * @example CreateRelationService::of('department')->from($user->getAttribute('services'))->getModels(),
 * @example CreateRelationService::of('department')->identifiedBy('code')->from($user->getAttribute('services'))->get(),
 * @example CreateRelationService::of('department')->from($user->getAttribute('services'))->withoutCreation->get(),
 * @example reference('department')->from($user->getAttribute('services'))->get(),
 * @example reference('department')->with('cities')->from($user->getAttribute('services'))->getModels(),
 * @example reference('department')->from($user->getAttribute('services'))->withoutCreation->only('fr'),
 */
final class CreateRelationService
{
    private BridgeContent $bridgeContent;

    private string $identifier = 'title';

    private string $mainLanguage;

    private Collection $additionalLanguages;

    private string | int | float | null $value;

    private string $operator = '=';

    private Collection $models;

    private bool $create = true;

    private bool $status = true;

    private array $relations = [];

    private function __construct(
    private string $contentType
  ) {
        $this->bridgeContent = new BridgeContent();

        $this->models = collect();

        $this->mainLanguage = config('app.locale');

        $this->additionalLanguages = collect(list_languages())->reject($this->mainLanguage);
    }

    /**
     * Permit to create the builder arround a specific content type
     *
     * @param string $contentType
     * @return self
     * @throws \Throwable
     */
    public static function of(string $contentType): self
    {
        $contentType = str($contentType)->snake()->trim();

        throw_unless(Schema::hasContentType($contentType), new Exception('Content-Type does not exist'));

        return new self($contentType);
    }

    public function withoutCreation(): self
    {
        $this->create = false;

        return $this;
    }

    /**
     * Permit to return an array of uuids for related content based on value from import
     *
     * @param string|int|float|null $value
     * @return $this
     */
    public function from(string | int | float | null $value = null): self
    {
        $this->value = match (true) {
            is_string($value) => empty($value) ? null : trim($value),
            default => $value
        };

        return $this;
    }

    /**
     * Permit to define the field, default is title
     *
     * @param string $field
     * @return $this
     * @throws \Throwable
     * @default title
     */
    public function identifiedBy(string $field): self
    {
        throw_unless(Schema::hasField($this->contentType, $field), new Exception('Field does not exist'));

        $this->identifier = $field;

        return $this;
    }

    /**
     * The query condition will use like operator
     * @return $this
     */
    public function usingLikeOperator(): self
    {
        $this->operator = 'like';

        return $this;
    }

    /**
     * Permit to define the status of the created content.
     * @default true
     * @param bool $condition
     * @return $this
     */
    public function publishedIf(bool $condition): self
    {
        $this->status = $condition;

        return $this;
    }

    /**
     * @param string $langcode
     * @return string|null
     */
    public function only(string $langcode): ?string
    {
        return $this->compute()->get($langcode)?->uuid;
    }

    /**
     * Get an array of uuids
     *
     * @return array
     */
    public function get(): array
    {
        return $this->compute()->values()->pluck('uuid')->all();
    }

    /**
     * Get the first uuid
     *
     * @return string|null
     */
    public function first(): ?string
    {
        return $this->compute()->values()->first()?->uuid;
    }

    /**
     * Get a collection of models
     *
     * @return Collection
     */
    public function getModels(): Collection
    {
        return $this->compute()->values();
    }

    /**
     * @param array|string $relations
     * @return $this
     */
    public function with(array|string $relations): self
    {
        $this->relations = is_string($relations) ? [$relations] : $relations;

        return $this;
    }

    /**
     * @return Collection
     */
    private function compute(): Collection
    {
        if (is_null($this->value)) {
            return collect();
        }

        $alreadyExists = $this->isContentExists($this->mainLanguage);

        if (! $alreadyExists && ! $this->create) {
            return collect();
        }

        $content = match ($alreadyExists) {
            true => $this->getContent($this->mainLanguage),
            false => $this->createContent($this->mainLanguage)
        };

        if ($this->contentType !== 'users' && Schema::getModelOptions($this->contentType)['translatable']) {
            $this->additionalLanguages->each(fn (string $langcode) => $this->translate($content, $langcode));
        }

        return $this->models;
    }

    /**
     * @return Builder
     */
    private function query(): Builder
    {
        return type_to_class($this->contentType)::query()->with($this->relations);
    }

    /**
     * Check if a content already exists for a specific language
     *
     * @param string $langcode
     * @return bool
     */
    private function isContentExists(string $langcode): bool
    {
        return $this->query()
      ->when($this->contentType !== 'users', fn (Builder $query) => $query->where('langcode', $langcode))
      ->when(
          value: $this->operator === 'like',
          callback: fn (Builder $query) => $query->whereLike($this->identifier, '%'.$this->value.'%'),
          default: fn (Builder $query) => $query->where($this->identifier, $this->operator, $this->value)
      )
      ->exists();
    }

    /**
     * Find content for a specific language and append the uuid to uuids list
     *
     * @param string $langcode
     * @return Content
     */
    private function getContent(string $langcode): Content
    {
        /** @var Content $content */
        $content = $this->query()
      ->when($this->contentType !== 'users', fn (Builder $query) => $query->where('langcode', $langcode))
      ->when(
          value: $this->operator === 'like',
          callback: fn (Builder $query) => $query->whereLike($this->identifier, '%'.$this->value.'%'),
          default: fn (Builder $query) => $query->where($this->identifier, $this->operator, $this->value)
      )
      ->first();

        return tap($content, fn () => $this->models->put($langcode, $content));
    }

    /**
     * Create a content throught the bridge for a specific language and append the uuid to uuids list
     *
     * @param string $langcode
     * @param string|null $uuid_host
     * @return Content
     */
    private function createContent(string $langcode, ?string $uuid_host = null): Content
    {
        $data = [
            'langcode' => $langcode,
            'status' => $this->status,
            $this->identifier => $this->value,
        ];

        if ($this->identifier !== 'title') {
            $data['title'] = $this->value;
        }

        if ($uuid_host) {
            $data['uuid_host'] = $uuid_host;
        }

        $uuid = $this->bridgeContent->contentInsert($this->contentType, $data, true, true);

        /** @var Content $content */
        $content = $this->query()->find($uuid);

        Log::channel('import')->info(
            str('[Import] Related Content Created, Content: ?, Identifier: ?, Value: ?')->replaceArray('?', [
                $this->contentType,
                $this->identifier,
                $this->value,
            ])
        );

        return tap($content, fn () => $this->models->put($langcode, $content));
    }

    /**
     * Find or create translated content and append it's uuid to uuids list.
     *
     * @param Content $content
     * @param string $langcode
     * @return void
     */
    private function translate(Content $content, string $langcode): void
    {
        /** @var Content $content */
        $translation = $content->getTranslationIfExists($langcode);

        if ($translation->langcode !== $langcode) {
            $translation = null;
        }

        $this->models->put($langcode, $translation ?? $this->createContent($langcode, $content->uuid_host));
    }
}
