<?php

namespace Inside\Import\Services;

use Carbon\Exceptions\InvalidFormatException;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Log;

/**
 * @example ConverteDateService::format('d/m/Y')->from('19/12/2023')->get()
 * @example ConverteDateService::format('d/m/Y')->from('19/12/2023')->birthday()
 * @example ConverteDateService::format('d/m/Y')->from('19/12/2023')->get()?->startOfDay()
 * @example carbonize('d/m/Y')->from('19/12/2023')->get()
 * @example carbonize('d/m/Y')->from('19/12/2023')->birthday()
 * @example carbonize('d/m/Y')->from('19/12/2023')->get()?->startOfDay()
 */
final class ConvertDateService
{
    private ?Carbon $value = null;

    private function __construct(protected string $format)
    {
        //
    }

    /**
     * @param string $format
     * @return self
     */
    public static function format(string $format): self
    {
        return new self($format);
    }

    /**
     * @param string|null $date
     * @return $this
     */
    public function from(?string $date): self
    {
        if (is_null($date)) {
            return $this;
        }

        try {
            $this->value = get_date($date, $this->format) ?: null;
        } catch (InvalidFormatException) {
            Log::channel('import')->info(
                str('[Import] Date "?" is not valid with this format ?')->replaceArray('?', [$date, $this->format])
            );
        }

        return $this;
    }

    /**
     * @return Carbon|null
     */
    public function get(): ?Carbon
    {
        return $this->value;
    }

    /**
     * @return string|null
     */
    public function birthday(): ?string
    {
        return $this->value?->startOfDay()->toDateTimeString();
    }
}
