<?php

namespace Inside\Import\Contracts;

use Inside\Host\Bridge\BridgeContent;

interface ImporterInterface
{
    /**
     * Function supposed to initialize everything for the import
     * @return bool
     */
    public function prepareImport(): bool;

    /**
     * Return user data after synchronized attribute with the type of import (syn_attributes in the config file)
     *
     * @param mixed $user
     * @param array $entry
     * @param BridgeContent $bridge
     * @return array
     */
    public function getSyncAttributes($user, array $entry, BridgeContent $bridge): array;

    /**
     * Get the function cleanData (launched after retrieving data for a given user) from config file
     * Signature must be : function(&$data)
     *
     * @return mixed
     */
    public function getCleanData();

    /**
     * Get all users from the base provider (google, csv file, etc.)
     * If $test is provided, create users from $test.
     *
     * @param array $entry
     * @param string|null $filter
     * @param string|null $test
     * @param int $limit
     * @return array
     */
    public function getUsers(array $entry, ?string $filter, ?string $test = null, int $limit = 0): array;

    /**
     * Get all entries from the base provider
     * (example: for google, one entry is a domain)
     * (example: for csv, one entry is one file)
     *
     * if there is no several entry, must return ['default' => entry] (something like that)
     * @return array
     */
    public function getEntries(): array;

    /**
     * Get the function preFlight (launched before the import) from the config file
     * Signature must be : function(Command|null $console, BridgeContent $bridge)
     *
     * @return mixed
     */
    public function getPreFlight();

    /**
     * Get the function postFlight (launched after the import) from config file
     * Signature must be : function(Command|null $console, BridgeContent $bridge)
     *
     * @return mixed
     */
    public function getPostFlight();

    /**
     * Get filters (from .env or config files, depends of import type)
     *
     * @return null|string
     */
    public function getFilter(): ?string;

    /**
     * Get User information from AD, null if user not found.
     * If $username is null, the first user of the AD is returned.
     */
    public function getUserInformation(?string $username, array $entry): ?array;
}
