<?php

namespace Inside\Import\Console;

use Inside\Host\Bridge\BridgeContent;
use Inside\Import\Contracts\ImporterInterface;
use Inside\Import\Services\UserReferencesCleaner;

class DumpCommand extends ImportCommand
{
    /**
     * @var string
     */
    protected $name = 'inside:user:dump';

    /*
   * @var string
   */
    protected $signature = 'inside:user:dump
                            { type? : Type of ad to be dumped (if empty, display allowed types) }
                            { username? : Name of the user we want to dump }
                            { --e|entry=* }
                            { --disable-postfilter }
                            { --disable-group-filter }
                           ';

    /**
     * @var string
     */
    protected $description = 'User import';

    public function __construct(private UserReferencesCleaner $userReferencesCleaner)
    {
        parent::__construct($userReferencesCleaner);
    }

    /**
     * Import users
     */
    public function handle(): void
    {
        /** @var string|null $type */
        $type = $this->argument('type');

        /** @var string|null $username */
        $username = $this->argument('username');

        /** @var array $entriesOption */
        $entriesOption = $this->option('entry');
        $disablePostfiter = (bool) $this->option('disable-postfilter');
        $disableGroupFilter = (bool) $this->option('disable-group-filter');
        $importer = $this->getImporter($type);

        if ($importer === null) {
            return;
        }

        $this->importer = $importer;
        $entries = $this->getEntries($entriesOption);

        if (empty($entries)) {
            $this->error('No entry found.');

            return;
        }

        $user = null;
        foreach ($entries as $name => $entry) {
            try {
                if (
                    isset($entry['postfilter']) &&
                    is_callable($entry['postfilter']) &&
                    $disablePostfiter
                ) {
                    config(['azuread.postfilter' => false]);
                    unset($entry['postfilter']);
                }

                if ($type === 'right') {
                    $user = (object) $this->importer->getUserInformation($username, $entry);
                } else {
                    if ($disableGroupFilter || $type !== 'azuread') {
                        $user = (object) $this->importer->getUserInformation($username, $entry);
                    } else {
                        $filter = ! empty($username) ? "mail eq '$username'" : '';
                        $users = $this->importer->getUsers($entry, $filter, null, 0);
                        $users = collect($users);
                        $user = $username ?
                            $users->filter(fn ($user) => $user->mail = $username)->first() :
                            collect($users)->first();
                    }
                }

                if (! empty($user) && (! empty($entry['custom_attributes']) || ! empty($entry['custom_security_attributes']))) {
                    $data = $this->importer->getSyncAttributes($user, $entry, new BridgeContent());

                    foreach ($data as $attributeKey => $attributeValue) {
                        if (! isset($user->$attributeKey)) {
                            $user->$attributeKey = json_encode($attributeValue).' (from custom_attributes)';
                        }
                    }
                }
            } catch (\Exception $exception) {
                $this->error($exception->getMessage());
            }
        }

        if (empty($user)) {
            $this->error("User [$username] not found.");

            return;
        }

        $rows = [];
        foreach ((array) $user as $key => $value) {
            $value = $this->transformValue($value);
            $rows[] = [$key, $value];
        }

        $this->table(['Clé', 'Valeur'], $rows);
    }

    /**
     * @param mixed $value
     */
    protected function transformValue($value): string
    {
        if (is_string($value)) {
            return $value;
        }

        if (is_null($value)) {
            return '(null)';
        }

        if (is_bool($value)) {
            return ($value ? 'true' : 'false').' (boolean)';
        }

        if (is_array($value) || is_object($value)) {
            $transformed = [];
            foreach ($value as $key => $elem) {
                $transformed[$key] = $this->transformValue($elem);
            }
            $formatted = [];
            foreach ($transformed as $key => $elem) {
                $formatted[] = "$key => $elem";
            }
            if (is_array($value)) {
                return '['.implode(",\n", $formatted).'] (array)';
            }

            return '{'.implode(",\n", $formatted).'} (object)';
        }

        return '<NA>';
    }

    protected function displayConfiguredTypes(): void
    {
        $types = array_keys(config('import.types', []));
        if (empty($types)) {
            $this->error('There is no import types configured');
        } else {
            $this->info('Configured import types:');
            foreach ($types as $type) {
                $this->info(" - $type");
            }
        }
    }
}
