<?php

namespace Inside\Statistics\Services;

use Carbon\Carbon;
use Exception;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Contracts\Pagination\Paginator;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Filesystem\FilesystemAdapter;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Inside\Authentication\Models\Token;
use Inside\Authentication\Models\User;
use Inside\Content\Contracts\Transformer;
use Inside\Content\Exceptions\ModelSchemaNotFoundException;
use Inside\Content\Facades\ContentHelper;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Content;
use Inside\Content\Models\Contents\Users;
use Inside\Content\Transformers\ContentTransformer;
use Inside\Database\Eloquent\Builder;
use Inside\Permission\Facades\Role;
use Inside\Statistics\Charts\GlobalChart;
use Inside\Statistics\Contracts\Statistic as StatisticContract;
use Inside\Statistics\Facades\Stats;
use Inside\Statistics\Models\GlobalStatistic;
use Inside\Statistics\Models\Statistic;
use Inside\Statistics\Repositories\StatisticsBetaRepository;
use Inside\Support\Str;
use Laravel\Lumen\Application;

class StatisticsService implements StatisticContract
{
    protected array $config;

    public function __construct(
    protected Application $app,
    protected Transformer $transformer,
    protected StatisticsBetaRepository $repository
  ) {
        $this->config = config('statistics.types', []);
    }

    public function getConfig(): array
    {
        return $this->config;
    }

    public function create(array $data, array $fields = []): Statistic
    {
        $statistic = Statistic::where($data)->first();
        if (is_null($statistic)) {
            $statistic = Statistic::create($data);
            Log::debug('Statistic successfully created', [
                'type' => $statistic->type,
                'uuid' => $statistic->user_uuid,
                'related_uuid' => $statistic->statisticable_uuid,
                'related_type' => $statistic->statisticable_type,
            ]);
        }
        $this->format($statistic, $fields);

        return $statistic;
    }

    public function format(Statistic $statistic, array $fields = []): array
    {
        return ['data' => $this->transformer->transform($statistic, $fields)];
    }

    public function countContent(array $types = [], array $filters = []): array
    {
        $statistics = [];
        $langCode = null;

        if (isset($filters['langcode'])) {
            $langCode = $filters['langcode'];
        }

        $statistics['likes'] = [
            'label' => Lang::get('reaction.type.like'),
            'count' => $this->repository->getReactionCountQuery('like', $langCode)->count(),
            'icon' => 'sid2-like',
        ];
        $user = Auth::user();

        foreach ($types as $type) {
            if (Schema::hasContentType($type)) {
                $options = Schema::getModelOptions($type);
                $query = call_user_func(type_to_class($type).'::query')->whereStatus(true);
                if ($langCode !== null) {
                    $query->whereLangcode($langCode);
                }
                $statistics[$type] = [
                    'label' => $options['title'][$user->langcode],
                    'count' => $query->count(),
                    'icon' => $options['content_type_icon_class'] ?? '',
                ];
            }
        }

        return $statistics;
    }

    public function getContentStatistics(string $type, array $filters = []): Paginator|\Illuminate\Database\Eloquent\Collection|LengthAwarePaginator|array
    {
        $query = $this->repository->getContentStatsQuery($type, $filters);
        ContentHelper::applySortToQuery($query, $filters, 'published_at', 'desc', [
            'title',
            'created_at',
            'published_at',
            'visits',
            'unique_visits',
            'updated_at',
            'status',
            'category_title',
            'likes',
            'comments',
        ]);

        return ContentHelper::getResultFromQuery($query, $filters);
    }

    public function getAuthenticationStats(array $filters = []): Paginator|\Illuminate\Database\Eloquent\Collection|LengthAwarePaginator|array
    {
        $query = $this->repository->getAuthenticationStatsQuery($filters);

        return tap(ContentHelper::getResultFromQuery($query, $filters, true, $query->get()->count()))->transform(function ($stat) {
            if ($stat instanceof User) {
                return [
                    'uuid' => $stat->uuid,
                    'full_name' => $stat->information != null ? $this->getFullName($stat->information) : '',
                    'email' => $stat->email,
                    'enabled' => $stat->status,
                    'roles' => $stat->permission->roles->pluck('name')->filter(function ($name) {
                        return ! Str::startsWith($name, ['workflow-', 'group-']);
                    })->map(function ($name) {
                        return Role::getHumanName($name);
                    })->implode(', '),
                    'connection_count' => 0,
                    'last_login_at_timestamp' => $stat->last_login_at->timestamp ?? null,
                ];
            }

            $roles = $stat['roles'];
            if (is_string($roles)) {
                $roles = explode(',', $roles);
            }
            if (is_null($roles)) {
                $roles = [];
            }
            if (! is_array($roles)) {
                $roles = [$roles];
            }
            foreach ($roles as $k => &$role) {
                if (Str::startsWith($role, ['workflow-', 'group-'])) {
                    unset($roles[$k]);
                    continue;
                }
                $role = Role::getHumanName($role);
            }

            $stat['roles'] = implode(', ', $roles);

            return $stat;
        });
    }

    public function getFullNameSelectRawQuery(): string
    {
        return "CONCAT(COALESCE(CONCAT(UPPER(LEFT(firstname, 1)), SUBSTRING(LOWER(firstname), 2)), ''), ' ', COALESCE(UPPER(lastname), ''))";
    }

    public function getFullName(Users $user): string
    {
        if (
            is_string($user->firstname)
            && ! empty($user->firstname)
            && is_string($user->lastname)
            && ! empty($user->lastname)
        ) {
            return Str::ucfirst(Str::lower($user->firstname)).' '.Str::upper($user->lastname);
        } elseif (is_string($user->firstname) && ! empty($user->firstname)) {
            return Str::ucfirst(Str::lower($user->firstname));
        } elseif (is_string($user->lastname) && ! empty($user->lastname)) {
            return Str::upper($user->lastname);
        } else {
            return '';
        }
    }

    public function getSocialStats(array $filters = []): array
    {
        return $this->repository->getSocialStats($filters);
    }

    public function getContentReaders(Content $content, array $filters, ?string $search = null): Paginator|\Illuminate\Database\Eloquent\Collection|LengthAwarePaginator|array
    {
        $notRead = $this->repository->notReadFiltered($filters);
        $query = $this->repository->getContentReadersQuery($content, $filters, $search);
        /** @var FilesystemAdapter $local */
        $local = Storage::disk('local');

        return tap(ContentHelper::getResultFromQuery($query, $filters, true))->transform(function ($user) use ($local) {
            if (! $user instanceof Users) {
                $user = $user->user;
            }

            return [
                'uuid' => $user->uuid,
                'full_name' => $this->getFullName($user),
                'email' => $user->email,
                'name' => $user->name,
                'image' => $this->getUserSecuredImage($local, $user),
            ];
        });
    }

    /**
     * @param FilesystemAdapter $local
     * @param Users $user
     * @return string
     */
    protected function getUserSecuredImage(FilesystemAdapter $local, Users $user): string
    {
        return str_replace(
            $local->url(''),
            '',
            $local->url(
                'files/'.$user->uuid.'/users/'.$user->image
            )
        );
    }

    public function getUserViewedContentList(User $user, array $types, array $filters, ?string $search = null): mixed
    {
        $contentsViewed = collect($types)->flatMap(function ($type) use ($user, $filters, $search) {
            return $this->repository->getUserViewedContentList($user, $type, $filters, $search)->get();
        });

        return ContentHelper::addPropertiesToQueryResult(
            [
                'viewed' => $contentsViewed->count(),
            ],
            ContentHelper::getPaginationFromCollection(
                collection: $contentsViewed,
                limit: $filters['limit'] ?? 10,
                page: (int) request()->input('page', 1),
                total: $contentsViewed->count(),
            )
        );
    }

    public function hasStatistics(string $type): bool
    {
        return ($type == 'users') || array_key_exists($type, $this->config);
    }

    public function addUniqueViewsToQuery(string $type, Builder|BelongsToMany $query): void
    {
        $query->addSelect([
            'unique_views' => function ($subquery) use ($type) {
                $subquery->select(DB::raw('COUNT(DISTINCT user_uuid)'))
                    ->from(type_to_stats_table($type))
                    ->whereColumn(type_to_stats_table($type).'.content_uuid', type_to_table($type).'.uuid');
            },
        ]);
    }

    public function addTotalViewsToQuery(string $type, Builder|BelongsToMany $query): void
    {
        $query->addSelect([
            'total_views' => function ($subquery) use ($type) {
                $subquery->select(DB::raw('COUNT(*)'))
                    ->from(type_to_stats_table($type))
                    ->whereColumn(type_to_stats_table($type).'.content_uuid', type_to_table($type).'.uuid');
            },
        ]);
    }

    public function getTypes(): Collection
    {
        $types = collect();
        foreach ($this->config as $type => $config) {
            try {
                $modelOptions = Schema::getModelOptions($type);
            } catch (ModelSchemaNotFoundException $exception) {
                continue;
            }
            $category = null;
            if ($config['category']) {
                try {
                    $categoryFieldOptions = Schema::getFieldOptions($type, $config['category']);
                    $category = [
                        'name' => Arr::first($categoryFieldOptions['target']),
                        'label' => $categoryFieldOptions['title'],
                    ];
                } catch (Exception $exception) {
                    Log::error('[StatisticsService::getTypes] failed to load options => '.$exception->getMessage());
                }
            }
            $types[] = [
                'name' => $type,
                'label' => $modelOptions['title'],
                'category' => $category,
                'has_likes' => $config['likes'] ?? false,
                'has_comments' => $config['comments'] ?? false,
            ];
        }

        return $types;
    }

    public function getGlobalChart(): GlobalChart
    {
        $chart = new GlobalChart();
        $data = GlobalStatistic::orderBy('date')
      ->groupBy([DB::raw('YEAR(date)'), DB::raw('MONTH(date)')])
      ->select('date')
      ->addSelect(DB::raw('SUM(likes) as likes'))
      ->addSelect(DB::raw('SUM(comments) as comments'));
        foreach (array_keys($this->config) as $type) {
            $data->addSelect(DB::raw('SUM('.$type.') as '.$type));
        }
        $chart->labels(
            $data->pluck('date')->transform(function ($date) {
                return get_date($date)?->format('m/Y');
            })
        );
        $chart->dataset('Likes', 'line', $data->pluck('likes'));
        $chart->dataset('Comments', 'line', $data->pluck('comments'));

        foreach (array_keys($this->config) as $type) {
            $chart->dataset((string) $type, 'line', $data->pluck($type));
        }

        return $chart;
    }

    public function getChartGlobalData(array $filters = []): array
    {
        $query = $this->repository->getChartGlobalData($filters);
        $dataSets = [
            ['name' => 'Likes', 'values' => $query->pluck('likes')],
            ['name' => 'Comments', 'values' => $query->pluck('comments')],
        ];
        foreach (array_keys($this->config) as $type) {
            $dataSets[] = ['name' => $type, 'values' => $query->pluck($type)];
        }

        return [
            'chart' => [
                'labels' => $query->pluck('date')->transform(function ($date) use ($filters) {
                    return get_date($date)?->format($this->getDateFormatFromFilter($filters));
                }),
            ],
            'datasets' => $dataSets,
        ];
    }

    public function getChartVisitsData(string $type, array $filters = []): array
    {
        $data = $this->repository->getChartVisitsData($type, $filters)->get()->keyBy('date');
        $start = get_date($data->first()->date);
        $end = get_date($data->last()->date);
        if (! $start) {
            return [];
        }
        for ($date = $start; $date < $end; $date = $this->getNextDateFromFilters($date, $filters)) {
            if (! $data->has($date->timestamp)) {
                $data[$date->timestamp] = [
                    'visits' => 0,
                    'unique_visits' => 0,
                    'likes' => 0,
                    'comments' => 0,
                    'date' => $date->timestamp,
                ];
            }
        }
        $data = $data->values()->sortBy('date');
        $dataSets = [
            ['name' => 'Visits', 'values' => $data->pluck('visits')],
            ['name' => 'Unique visits', 'values' => $data->pluck('unique_visits')],
            ['name' => 'Likes', 'values' => $data->pluck('likes')],
            ['name' => 'Comments', 'values' => $data->pluck('comments')],
        ];

        return [
            'chart' => [
                'labels' => $data->pluck('date')->transform(function ($date) use ($filters) {
                    return get_date($date)?->format($this->getDateFormatFromFilter($filters));
                }),
            ],
            'datasets' => $dataSets,
        ];
    }

    public function getChartUserVisitsData(string $type, User $user, array $filters = []): array
    {
        $data = $this->repository->getChartUserVisitsData($type, $user, $filters)->get()->keyBy('date');
        if ($data->isNotEmpty()) {
            $start = get_date($data->first()->date);
            $end = get_date($data->last()->date);
            if (! $start) {
                return [];
            }
            for ($date = $start; $date < $end; $date = $this->getNextDateFromFilters($date, $filters)) {
                if (! $data->has($date->timestamp)) {
                    $data[$date->timestamp] = [
                        'visits' => 0,
                        'unique_visits' => 0,
                        'date' => $date->timestamp,
                    ];
                }
            }
            $data = $data->values()->sortBy('date');
        }
        $dataSets = [
            ['name' => 'Visits', 'values' => $data->pluck('visits')],
            ['name' => 'Unique visits', 'values' => $data->pluck('unique_visits')],
        ];

        return [
            'chart' => [
                'labels' => $data->pluck('date')->transform(function ($date) use ($filters) {
                    return get_date($date)?->format($this->getDateFormatFromFilter($filters));
                }),
            ],
            'datasets' => $dataSets,
        ];
    }

    public function getChartAuthenticationsData(array $filters = []): array
    {
        $data = $this->repository->getChartAuthenticationsData($filters)->get()->keyBy('date');
        $start = get_date($data->first()->date);
        $end = get_date($data->last()->date);
        if (! $start) {
            return [];
        }
        for ($date = $start; $date < $end; $date = $this->getNextDateFromFilters($date, $filters)) {
            if (! $data->has($date->timestamp)) {
                $data[$date->timestamp] = [
                    'authentications' => 0,
                    'unique_authentications' => 0,
                    'date' => $date->timestamp,
                ];
            }
        }

        $data = $data->values()->sortBy('date');
        $dataSets = [
            ['name' => 'Connexions', 'values' => $data->pluck('authentications')],
            ['name' => 'Connexions uniques', 'values' => $data->pluck('unique_authentications')],
        ];

        return [
            'chart' => [
                'labels' => $data->pluck('date')->transform(function ($date) use ($filters) {
                    return get_date($date)?->format($this->getDateFormatFromFilter($filters));
                }),
            ],
            'datasets' => $dataSets,
        ];
    }

    public function getUserVisitStatistics(string $type, array $filters = [], ?string $search = null): mixed
    {
        return ContentHelper::getResultFromQuery(
            $this->repository->getUserVisitStatsQuery($type, $filters, $search),
            $filters
        );
    }

    protected function getDateFormatFromFilter(array $filters = []): string
    {
        if (isset($filters['mode']) && $filters['mode'] == 'month') {
            return 'm/Y';
        } elseif (isset($filters['mode']) && $filters['mode'] == 'year') {
            return 'Y';
        }

        return 'd/m/Y';
    }

    protected function getNextDateFromFilters(Carbon $date, array $filters = []): Carbon
    {
        if (isset($filters['mode']) && $filters['mode'] == 'month') {
            return $date->addMonth();
        } elseif (isset($filters['mode']) && $filters['mode'] == 'year') {
            return $date->addYear();
        }

        return $date->addDay();
    }
}
