<?php

namespace Inside\Statistics\Repositories;

use Illuminate\Auth\AuthenticationException;
use Illuminate\Database\Query\Builder as QueryBuilder;
use Illuminate\Support\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Schema as LaravelSchema;
use Illuminate\Support\Str;
use Inside\Authentication\Models\Token;
use Inside\Authentication\Models\User;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Content;
use Inside\Content\Models\Contents\Comments;
use Inside\Content\Models\Contents\Users;
use Inside\Content\Models\Section;
use Inside\Content\Transformers\ContentTransformer;
use Inside\Database\Eloquent\Builder;
use Inside\Reaction\Models\Reaction;
use Inside\Statistics\Facades\Stats;
use Inside\Statistics\Models\AuthenticationStatistic;
use Inside\Statistics\Models\AuthenticationStatistics;
use Inside\Statistics\Models\ContentStatistic;
use Inside\Statistics\Models\ContentStatistics;
use Inside\Statistics\Models\GlobalStatistic;
use Inside\Statistics\Models\UserStatistic;

class StatisticsBetaRepository
{
    public function getUserViewedContentList(User|Users $user, string $type, array $filters, ?string $search = null): Builder
    {
        [$fromDate, $toDate] = $this->extractDateFilters($filters);

        $query = ContentStatistics::bind($type)
            ->where('user_uuid', $user->uuid)
            ->with('content')
            ->select('content_uuid')
            ->when($fromDate && $toDate, fn ($query) => $query->whereBetween('date', [$fromDate, $toDate]));

        $query->whereHas('content', function ($query) use ($search) {
            $query->where(function ($query) use ($search) {
                if ($search !== null) {
                    $query->where('title', 'LIKE', "%$search%");
                }
            });
        });

        return $query->groupBy('content_uuid');
    }

    public function getSocialStats(array $filters = []): array
    {
        $statistics = [];
        $statistics['likes'] = [
            'label' => Lang::get('reaction.type.like'),
            'count' => (int) $this->getReactionCountQuery('like', $filters['langcode'] ?? null, $filters)->count(),
            'icon' => 'sid2-like',
        ];
        $commentsCountQuery = Comments::query()->where('status', '=', true);
        [$fromDate, $toDate] = $this->extractDateFilters($filters);

        if ($fromDate !== null) {
            $commentsCountQuery->where('created_at', '>=', $fromDate->toDateString());
        }
        if ($toDate !== null) {
            $commentsCountQuery->where('created_at', '<=', $toDate->toDateString());
        }
        $commentsCount = (int) $commentsCountQuery->count();
        $statistics['comments'] = [
            'label' => $commentsCount > 1 ? Lang::get('statistics.social.comments') : Lang::get('statistics.social.comment'),
            'count' => $commentsCount,
            'icon' => 'sid2-comment',
        ];

        return $statistics;
    }

    public function countContent(array $types = [], array $filters = []): array
    {
        $langCode = $filters['langcode'] ?? null;
        $likesStatistics = [
            'label' => Lang::get('reaction.type.like'),
            'count' => $this->getReactionCountQuery('like', $langCode)->count(),
            'icon' => 'sid2-like',
        ];
        $user = Auth::user();
        if ($user == null) {
            throw new AuthenticationException('Unauthenticated.');
        }
        $typesCollection = collect($types);
        $statistics = $typesCollection->filter(function ($type) {
            return Schema::hasContentType($type);
        })
            ->mapWithKeys(function ($type) use ($user, $langCode) {
                $options = Schema::getModelOptions($type);
                $query = call_user_func(type_to_class($type).'::query')->whereStatus(true);
                $query->where('published_at', '<=', Carbon::now()->toDateString());
                if ($langCode !== null) {
                    $query->whereLangcode($langCode);
                }

                return [
                    $type => [
                        'label' => $options['title'][$user->langcode],
                        'count' => $query->count(),
                        'icon' => $options['content_type_icon_class'] ?? '',
                    ],
                ];
            })
            ->toArray();
        $statistics['likes'] = $likesStatistics;

        foreach ($statistics as $key => $statistic) {
            $statistics[$key]['label'] = Str::plural($statistic['label']);
        }

        return $statistics;
    }

    public function getReactionCountQuery(string $type, ?string $langCode = null, array $filters = []): Builder
    {
        $query = Reaction::whereType($type)
            ->where(function ($query) use ($langCode) {
                foreach (Schema::getContentTypes() as $type) {
                    $query->orWhere(function ($query) use ($type, $langCode) {
                        if (Str::endsWith($type, 'menus')) {
                            return;
                        }
                        $query->where('inside_reactions.reactionable_type', type_to_class($type))
                            ->whereExists(function ($query) use ($type, $langCode) {
                                $query->select(type_to_table($type).'.*')
                                    ->from(type_to_table($type))
                                    ->whereStatus(true)
                                    ->whereColumn('uuid', 'inside_reactions.reactionable_uuid');
                                if ($langCode !== null) {
                                    $query->where(type_to_table($type).'.langcode', $langCode);
                                }
                            });
                    });
                }
            });
        [$fromDate, $toDate] = $this->extractDateFilters($filters);
        if ($fromDate !== null) {
            $query->where('created_at', '>=', $fromDate->toDateString());
        }
        if ($toDate !== null) {
            $query->where('created_at', '<=', $toDate->toDateString());
        }

        return $query;
    }

    public function getAuthenticationStatsQuery(array $filters = []): QueryBuilder
    {
        [$fromDate, $toDate] = $this->extractDateFilters($filters);
        [$order , $direction] = $this->getSortFilter($filters, 'last_login_at', 'desc');

        $order = match ($order) {
            'full_name' => 'inside_content_users.firstname',
            'email' => 'inside_content_users.email',
            default => $order,
        };

        $search = $filters['query'] ?? null;

        $statsSubquery = AuthenticationStatistics::selectRaw('SUM(connections) as connections, user_uuid')
            ->when($fromDate !== null, fn ($query) => $query->where('created_at', '>=', $fromDate->toDateString()))
            ->when($toDate !== null, fn ($query) => $query->where('created_at', '<=', $toDate->toDateString()))
            ->groupBy('user_uuid');

        $query = DB::table('inside_content_users')
            ->selectRaw('0 as connections')
            ->selectRaw('inside_content_users.uuid as user_uuid');

        if (isset($filters['never_authenticated'])) {
            $query = $this->getAuthenticatedQuery($query, $statsSubquery, $filters, $fromDate, $toDate);
        }
        if (isset($filters['never_access'])) {
            $query = $this->getAccessQuery($query, $statsSubquery, $filters, $fromDate, $toDate);
        }

        $query
            ->join('inside_users', 'inside_content_users.uuid', '=', 'inside_users.uuid')
            ->selectRaw("CONCAT(COALESCE(CONCAT(UPPER(LEFT(firstname, 1)), SUBSTRING(LOWER(firstname), 2)), ''), ' ', COALESCE(UPPER(lastname), '')) AS full_name");

        if ($search !== null) {
            $query->where(function ($query) use ($search) {
                $query->whereRawLike(
                    "CONCAT(COALESCE(CONCAT(UPPER(LEFT(firstname, 1)), SUBSTRING(LOWER(firstname), 2)), ''), ' ', COALESCE(UPPER(lastname), ''))",
                    "%$search%"
                )
                    ->orWhereLike('inside_content_users.email', "%$search%");
            });
        }

        return $query->orderBy($order, $direction);
    }

    private function getDateFilteredQuery(
        QueryBuilder $query,
        Builder $statsSubquery,
        array $filters,
        ?Carbon $fromDate,
        ?Carbon $toDate,
        string $dateField,
        string $neverFilterMethod
    ): QueryBuilder {
        $neverFilter = $this->{$neverFilterMethod}($filters);

        $query->when($neverFilter, function ($query) use ($statsSubquery, $fromDate, $toDate, $dateField) {
            $query->where('inside_content_users.status', true);
            $query->whereNotIn('inside_content_users.uuid', $statsSubquery->pluck('user_uuid'));

            $query->where(function ($query) use ($fromDate, $toDate, $dateField) {
                if ($fromDate !== null) {
                    $query->whereDate("inside_users.$dateField", '<', $fromDate);
                }
                if ($toDate !== null) {
                    $query->orWhereDate("inside_users.$dateField", '>', $toDate);
                }
                $query->orWhereNull("inside_users.$dateField");
            });
        });

        $query->when(! $neverFilter, function ($query) use ($fromDate, $toDate, $dateField) {
            $query->whereNotNull("inside_users.$dateField");

            if ($fromDate !== null) {
                $query->whereDate("inside_users.$dateField", '>=', $fromDate);
            }
            if ($toDate !== null) {
                $query->whereDate("inside_users.$dateField", '<=', $toDate);
            }
        });

        return $query;
    }

    private function getAuthenticatedQuery(QueryBuilder $query, Builder $statsSubquery, array $filters, ?Carbon $fromDate = null, ?Carbon $toDate = null): QueryBuilder
    {
        return $this->getDateFilteredQuery(
            $query,
            $statsSubquery,
            $filters,
            $fromDate,
            $toDate,
            'last_login_at',
            'neverAuthenticatedOnlyFiltered'
        );
    }

    private function getAccessQuery(QueryBuilder $query, Builder $statsSubquery, array $filters, ?Carbon $fromDate = null, ?Carbon $toDate = null): QueryBuilder
    {
        return $this->getDateFilteredQuery(
            $query,
            $statsSubquery,
            $filters,
            $fromDate,
            $toDate,
            'last_access_at',
            'neverAccessOnlyFiltered'
        );
    }

    public function neverAuthenticatedOnlyFiltered(array $filters = []): bool
    {
        return filter_var($filters['never_authenticated'] ?? false, FILTER_VALIDATE_BOOLEAN);
    }

    public function neverAccessOnlyFiltered(array $filters = []): bool
    {
        return filter_var($filters['never_access'] ?? false, FILTER_VALIDATE_BOOLEAN);
    }

    public function getUserVisitStatsQuery(string $type, array $filters = [], ?string $search = null): Builder
    {
        [$fromDate, $toDate] = $this->extractDateFilters($filters);

        $statsSubquery = ContentStatistics::bind($type)
            ->selectRaw('COUNT(*) as total_views')
            ->selectRaw('COUNT(DISTINCT content_uuid) as unique_views')
            ->selectRaw('user_uuid')
            ->whereExists(fn ($query) => $query
                ->select(DB::raw(1))
                ->from('inside_content_users')
                ->whereColumn(type_to_stats_table($type).'.user_uuid', 'inside_content_users.uuid')
            )
            ->when($fromDate !== null, fn ($query) => $query->startDate($fromDate->toDateString()))
            ->when($toDate !== null, fn ($query) => $query->endDate($toDate->toDateString()));

        $statsSubquery = $statsSubquery->groupBy('user_uuid');

        $query = Users::query()
            ->join(
                DB::raw("({$statsSubquery->toSql()}) as stats"),
                fn ($join) => $join->on('stats.user_uuid', '=', 'inside_content_users.uuid')
            )
            ->mergeBindings($statsSubquery->getQuery())
            ->selectRaw('stats.total_views')
            ->selectRaw('stats.unique_views')
            ->selectRaw('stats.user_uuid')
            ->selectRaw('inside_content_users.email')
            ->selectRaw('CONCAT(CONCAT(UPPER(LEFT(firstname, 1)), SUBSTRING(LOWER(firstname), 2)), CONCAT(\' \', UPPER(lastname))) AS full_name')
            ->when($search !== null, fn ($query) => $query
                ->whereRawLike("CONCAT(COALESCE(CONCAT(UPPER(LEFT(firstname, 1)), SUBSTRING(LOWER(firstname), 2)), ''), ' ', COALESCE(UPPER(lastname), ''))", "%$search%")
                ->orWhereLike('inside_content_users.email', "%$search%")
            );

        [$order , $direction] = $this->getSortFilter($filters, 'full_name', 'desc');

        return $query->orderBy($order, $direction);
    }

    public function getContentReadersQuery(Content | Section $content, array $filters, ?string $search = null): Builder
    {
        $query = ContentStatistics::bind($content->contentType)
            ->where('content_uuid', $content->uuid)
            ->with([
                'user' => fn ($query) => $query
                    ->select('uuid', 'firstname', 'lastname', 'image', 'email', 'status')
                    ->selectRaw("CONCAT(COALESCE(CONCAT(UPPER(LEFT(firstname, 1)), SUBSTRING(LOWER(firstname), 2)), ''), ' ', COALESCE(UPPER(lastname), '')) AS full_name"),
            ])
            ->select('user_uuid')
            ->groupBy('user_uuid');
        [$fromDate, $toDate] = $this->extractDateFilters($filters);

        if ($fromDate !== null) {
            $query->startDate($fromDate->toDateString());
        }
        if ($toDate !== null) {
            $query->endDate($toDate->toDateString());
        }
        if (isset($filters['not_read']) && $filters['not_read']) {
            $query = Users::whereNotIn('uuid', $query->get()->pluck('user.uuid')->toArray())
                ->select('uuid', 'firstname', 'lastname', 'image', 'email', 'status')
                ->selectRaw("CONCAT(COALESCE(CONCAT(UPPER(LEFT(firstname, 1)), SUBSTRING(LOWER(firstname), 2)), ''), ' ', COALESCE(UPPER(lastname), '')) AS full_name");
            if ($search !== null) {
                $query->where('status', true)->where(function ($subQuery) use ($search) {
                    $subQuery->whereRaw("CONCAT(COALESCE(CONCAT(UPPER(LEFT(firstname, 1)), SUBSTRING(LOWER(firstname), 2)), ''), ' ', COALESCE(UPPER(lastname), '')) LIKE ?", ["%$search%"])
                        ->orWhere('email', 'LIKE', "%$search%");
                });
            }

            return $query;
        } else {
            $query->whereHas('user', function ($query) use ($search) {
                if ($search !== null) {
                    $query->where('status', true)->where(function ($query) use ($search) {
                        $query->whereRaw("CONCAT(COALESCE(CONCAT(UPPER(LEFT(firstname, 1)), SUBSTRING(LOWER(firstname), 2)), ''), ' ', COALESCE(UPPER(lastname), '')) LIKE ?", ["%$search%"])
                            ->orWhere('email', 'LIKE', "%$search%");
                    });
                }
            });
        }

        return $query;
    }

    public function getContentStatsQuery(string $type, array $filters = []): Builder
    {
        $config = config('statistics.types.'.$type, []);
        [$fromDate, $toDate] = $this->extractDateFilters($filters);
        $categoryField = isset($config['category']) ? Str::camel($config['category']) : null;
        $queryWith = collect([
            'content' => fn ($query) => $query->select('uuid', 'title', 'status', 'created_at', 'langcode', 'published_at', 'updated_at'),
        ]);
        if ($categoryField) {
            $queryWith->put('content.'.$categoryField, fn ($query) => $query->select('title', 'uuid'));
        }
        if ($config['comments']) {
            $queryWith->put('content.comments', fn ($query) => $query
                ->select('id', 'parent_uuid')
                ->when(! blank($fromDate) && ! blank($toDate), fn ($query) => $query->whereBetween('created_at', [$fromDate, $toDate]))
            );
        }

        $query = ContentStatistics::bind($type)
            ->selectRaw('COUNT(user_uuid) as total_views')
            ->selectRaw('COUNT(DISTINCT(user_uuid)) as unique_views, content_uuid')
            ->selectRaw(type_to_table($type).'.status')
            ->join(type_to_table($type), type_to_stats_table($type).'.content_uuid', '=', type_to_table($type).'.uuid')
            ->leftJoin('inside_content_users', 'user_uuid', '=', 'inside_content_users.uuid')
            ->when(blank($fromDate) || blank($toDate),
                fn ($query) => $query->selectRaw("(SELECT COUNT(*) FROM inside_reactions WHERE reactionable_uuid = content_uuid AND type = 'like') as likes"),
                fn ($query) => $query->selectRaw("(SELECT COUNT(*) FROM inside_reactions WHERE reactionable_uuid = content_uuid AND type = 'like' AND created_at BETWEEN '$fromDate' AND '$toDate') as likes")
            )
            ->with($queryWith->toArray());

        if (LaravelSchema::hasTable('inside_archives')) {
            $query->selectRaw("(SELECT EXISTS(SELECT 1 FROM inside_archives WHERE uuid = content_uuid AND date <= '".date('Y-m-d H:i:s')."')) as archived");
        }

        $this->applySortToQuery($query, $filters, 'published_at', 'desc', $type);

        if ($fromDate !== null && $toDate !== null) {
            $query->whereBetween(type_to_stats_table($type).'.date', [$fromDate, $toDate]);
        }

        if (isset($filters['langcode'])) {
            $langcode = $filters['langcode'];
            $query->whereHas('content', function ($query) use ($langcode) {
                $query->where('langcode', $langcode);
            });
        }
        if ($categoryField && isset($filters['category_uuid'])) {
            $categoryUuids = is_array($filters['category_uuid']) ? $filters['category_uuid'] : [$filters['category_uuid']];
            $query->whereHas('content.'.Str::camel($config['category']), function ($query) use ($categoryUuids) {
                $query->whereIn('uuid', $categoryUuids);
            });
        }

        return $query->groupBy(['content_uuid', type_to_table($type).'.status', type_to_table($type).'.published_at']);
    }

    public function getSortFilter(array $filters, string $defaultOrder, string $defaultDirection): array
    {
        $sort = $filters['sort'] ?? $defaultOrder;
        if (strpos($sort, ':') !== false) {
            [$order, $direction] = explode(':', $sort, 2);
            if (! in_array(Str::lower($direction), ['asc', 'desc'])) {
                $direction = 'asc';
            }
        }
        $direction = $direction ?? $defaultDirection;
        $order = $order ?? $defaultOrder;

        return [$order, $direction];
    }

    private function applySortToQuery(Builder $query, array $filters, string $defaultOrder, string $defaultDirection = 'asc', string $type = null): void
    {
        [$order , $direction] = $this->getSortFilter($filters, $defaultOrder, $defaultDirection);
        $config = config('statistics.types.'.$type, []);
        if ($config['category'] === $order || $order === 'comments') {
            $query->orderBy(function ($query) use ($type, $order) {
                $table = type_to_table($order);
                $query = $order === 'comments' ? $query->selectRaw('COUNT('.$table.'.uuid)') : $query->select($table.'.title');
                $query->from($table)
                    ->join('inside_pivots', $table.'.uuid', '=', 'inside_pivots.related_uuid')
                    ->whereColumn('inside_pivots.parent_uuid', type_to_stats_table($type).'.content_uuid')
                    ->limit(1);
            }, $direction);
        } else {
            $query->orderBy($order, $direction);
        }
    }

    private function extractDateFilters(array $filters): array
    {
        $fromDate = $this->getDateFromFilters($filters, 'from');
        $toDate = $this->getDateFromFilters($filters, 'to');

        return [$fromDate, $toDate];
    }

    protected function getDateFromFilters(array $filters, string $filterName): ?Carbon
    {
        $fromDate = null;
        if (isset($filters[$filterName])) {
            $fromDate = $this->getCarbonFromFront($filters[$filterName]);
        }

        return $fromDate;
    }

    public function getCarbonFromFront(string $date): ?Carbon
    {
        try {
            return get_date($date, 'Y-m-d')->setTime(12, 0);
        } catch (\Exception $exception) {
            Log::error('[StatisticsRepository::getCarbonFromFront] Failed to parse date from from => '.$exception->getMessage());
        }

        return null;
    }

    public function notReadFiltered(array $filters = []): bool
    {
        return ! isset($filters['not_read'])
            || $filters['not_read'] === true
            || in_array($filters['not_read'], [1, '1', 'true']);
    }

    public function getChartGlobalData(array $filters = []): Builder
    {
        $fromDate = $this->getDateFromFilters($filters, 'from');
        $toDate = $this->getDateFromFilters($filters, 'to');
        $query = GlobalStatistic::orderBy('date')
            ->groupBy($this->getGroupByFromFilters($filters))
            ->select('date')
            ->addSelect(DB::raw('SUM(likes) as likes'))
            ->addSelect(DB::raw('SUM(comments) as comments'));
        foreach (array_keys(Stats::getConfig()) as $type) {
            $query->addSelect(DB::raw('SUM('.$type.') as '.$type));
        }

        if ($fromDate !== null) {
            $query->whereDate('global_statistics.date', '>=', $fromDate->toDateString());
        }
        if ($toDate !== null) {
            $query->whereDate('global_statistics.date', '<=', $toDate->toDateString());
        }

        return $query;
    }

    public function getChartVisitsData(string $type, array $filters = []): Builder
    {
        $fromDate = $this->getDateFromFilters($filters, 'from');
        $toDate = $this->getDateFromFilters($filters, 'to');
        $query = ContentStatistic::orderBy('date')
            ->groupBy(
                $this->getGroupByFromFilters($filters)
            )->select('date')
            ->addSelect(DB::raw('SUM(likes) as likes'))
            ->addSelect(DB::raw('SUM(comments) as comments'))
            ->addSelect(DB::raw('SUM(visits) as visits'))
            ->addSelect(DB::raw('SUM(unique_visits) as unique_visits'))
            ->where('type', $type);

        if ($fromDate !== null) {
            $query->whereDate('content_statistics.date', '>=', $fromDate->toDateString());
        }
        if ($toDate !== null) {
            $query->whereDate('content_statistics.date', '<=', $toDate->toDateString());
        }

        return $query;
    }

    public function getChartUserVisitsData(string $type, User $user, array $filters = []): Builder
    {
        $fromDate = $this->getDateFromFilters($filters, 'from');
        $toDate = $this->getDateFromFilters($filters, 'to');
        $query = UserStatistic::orderBy('date')
            ->groupBy($this->getGroupByFromFilters($filters))
            ->select('date')
            ->addSelect(DB::raw('SUM(visits) as visits'))
            ->addSelect(DB::raw('SUM(unique_visits) as unique_visits'))
            ->where('type', $type)
            ->where('uuid', $user->uuid);

        if ($fromDate !== null) {
            $query->whereDate('user_statistics.date', '>=', $fromDate->toDateString());
        }
        if ($toDate !== null) {
            $query->whereDate('user_statistics.date', '<=', $toDate->toDateString());
        }

        return $query;
    }

    public function getChartAuthenticationsData(array $filters = []): Builder
    {
        $fromDate = $this->getDateFromFilters($filters, 'from');
        $toDate = $this->getDateFromFilters($filters, 'to');
        $query = AuthenticationStatistic::orderBy('date')
            ->groupBy($this->getGroupByFromFilters($filters))
            ->select('date')
            ->addSelect(DB::raw('COUNT(uuid) as unique_authentications'))
            ->addSelect(DB::raw('SUM(connection_count) as authentications'));

        if ($fromDate !== null) {
            $query->whereDate('authentication_statistics.date', '>=', $fromDate->toDateString());
        }
        if ($toDate !== null) {
            $query->whereDate('authentication_statistics.date', '<=', $toDate->toDateString());
        }

        return $query;
    }

    protected function getGroupByFromFilters(array $filters = []): array
    {
        if (isset($filters['mode']) && $filters['mode'] == 'month') {
            return [DB::raw('YEAR(date)'), DB::raw('MONTH(date)')];
        } elseif (isset($filters['mode']) && $filters['mode'] == 'year') {
            return [DB::raw('YEAR(date)')];
        }

        return ['date'];
    }
}
