<?php

namespace Inside\Statistics\Jobs;

use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Inside\Content\Models\Contents\Comments;
use Inside\Content\Models\Pivot;
use Inside\Jobs\Job;
use Inside\Reaction\Models\Reaction;
use Inside\Statistics\Models\Statistic;
use Symfony\Component\Console\Output\OutputInterface;

class ComputeGlobalStatistics extends Job
{
    protected array $config;

    /**
     * @var int<1, max>
     */
    protected int $chunk = 1000;

    public function __construct(
        protected ?OutputInterface $output = null
    ) {
    }

    public function handle(): void
    {
        $this->config = config('statistics.types');

        $this->checkGlobalStatisticsTable();
        DB::table('global_statistics')->truncate();
        $stats = [];
        $this->computeReactions('like', 'likes', $stats);
        $this->computeContentStats(
            Comments::whereHas('authors', function ($query) {
                $query->where('is_maintenance', false);
            })->whereDate('created_at', '<', now()->toDateString())->get(),
            'comments',
            $stats
        );

        foreach (array_keys($this->config) as $type) {
            $query = call_user_func(type_to_class($type).'::query');
            $contents = $query->whereDate('created_at', '<', now()->toDateString())->get();
            $this->computeContentStats($contents, $type, $stats);
        }
        $bar = null;
        if ($this->output) {
            $bar = $this->output->createProgressBar(count($stats));
            $bar->start();
        }
        foreach (array_chunk($stats, $this->chunk) as $chunked) {
            DB::table('global_statistics')->insert($chunked);
            if ($bar) {
                $bar->advance(count($chunked));
            }
        }

        if ($bar) {
            $bar->finish();
            if ($this->output instanceof OutputInterface) {
                $this->output->writeln('');
                $this->output->writeln('<fg=cyan>'.str_repeat('#', 80).'</>');
            }
        }
    }

    /**
     * @todo cleanup removed types
     */
    protected function checkGlobalStatisticsTable(): void
    {
        foreach (array_keys($this->config) as $type) {
            if (! Schema::hasColumn('global_statistics', $type)) {
                Schema::table('global_statistics', function (Blueprint $table) use ($type) {
                    $table->integer($type)->default(0);
                });
            }
        }
    }

    /**
     * @param string $type
     * @param string $name
     * @param array $stats
     * @return void
     */
    protected function computeReactions(string $type, string $name, array &$stats): void
    {
        $reactions = Reaction::whereHas('user', function ($query) {
            $query->where('is_maintenance', false);
        })->where('type', $type)
            ->whereDate('created_at', '<', now()->toDateString())->get();
        $bar = null;
        if ($this->output) {
            $bar = $this->output->createProgressBar($reactions->count());
            $bar->start();
        }
        foreach ($reactions as $reaction) {
            $date = get_date($reaction->created_at)->setTime(12, 0);
            if (! array_key_exists($date->toDateString(), $stats)) {
                $stats[$date->toDateString()] = $this->initGlobalStat($date);
            }
            $stats[$date->toDateString()][$name]++;
            if ($bar) {
                $bar->advance();
            }
        }
        if ($bar) {
            $bar->finish();
        }
    }

    /**
     * @param Carbon $date
     * @return array
     */
    private function initGlobalStat(Carbon $date): array
    {
        $statistic = [
            'likes'    => 0,
            'comments' => 0,
            'date'     => $date->toDateString(),
        ];
        foreach (array_keys($this->config) as $type) {
            $statistic[$type] = 0;
        }

        return $statistic;
    }

    /**
     * @param Collection $contents
     * @param string $type
     * @param array $stats
     */
    protected function computeContentStats(Collection $contents, string $type, array &$stats): void
    {
        $bar = null;
        if ($this->output) {
            $bar = $this->output->createProgressBar($contents->count());
            $bar->start();
        }
        foreach ($contents as $content) {
            $date = get_date($content->created_at)->setTime(12, 0);
            if (! array_key_exists($date->toDateString(), $stats)) {
                $stats[$date->toDateString()] = $this->initGlobalStat($date);
            }
            $stats[$date->toDateString()][$type]++;
            if ($bar) {
                $bar->advance();
            }
        }
        if ($bar) {
            $bar->finish();
        }
    }

    /**
     * @param string $type
     * @param string $name
     * @param array  $stats
     * @return void
     */
    protected function computeStats(string $type, string $name, array &$stats): void
    {
        $query = Statistic::where('type', $type)
            ->whereDate('created_at', '<', now()->toDateString());
        $bar = null;
        if ($this->output) {
            $bar = $this->output->createProgressBar($query->count());
            $bar->start();
        }
        $query->each(function (Statistic $statistic) use (&$stats, $name, $bar) {
            if ($bar) {
                $bar->advance();
            }
            if ($statistic->statisticable === null) {
                return;
            }
            if ($statistic->user === null
                || $statistic->user->information === null || $statistic->user->information->is_maintenance
            ) {
                return;
            }
            $contentType = null;
            if ($statistic->statisticable instanceof Reaction) {
                $contentType = class_to_type($statistic->statisticable->reactionable);
            } elseif ($statistic->statisticable instanceof Comments) {
                $content = DB::table('inside_pivots')->select('parent_type')->where('related_type', Comments::class)
                             ->where('related_uuid', $statistic->statisticable_uuid)->where(
                                 'related_field',
                                 'comments'
                             )->first();
                if ($content !== null && isset($content->parent_type)) {
                    $contentType = class_to_type($content->parent_type);
                }
            }
            if ($contentType === null || ! array_key_exists($contentType, $this->config)) {
                return;
            }

            $date = get_date((int) $statistic->created_at)->setTime(12, 0);
            if (! array_key_exists($date->toDateString(), $stats)) {
                $stats[$date->toDateString()] = $this->initGlobalStat($date);
            }
            $stats[$date->toDateString()][$name]++;
        });
        $bar->finish();
    }
}
