<?php

namespace Inside\Statistics\Exports;

use Illuminate\Database\Query\Builder;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Lang;
use Inside\Authentication\Models\User;
use Inside\Content\Facades\ContentHelper;
use Inside\Permission\Facades\Role;
use Inside\Statistics\Facades\Stats;
use Inside\Statistics\Repositories\StatisticsBetaRepository;
use Inside\Support\Str;
use Maatwebsite\Excel\Concerns\WithColumnFormatting;
use Maatwebsite\Excel\Concerns\WithStrictNullComparison;
use PhpOffice\PhpSpreadsheet\Shared\Date;
use PhpOffice\PhpSpreadsheet\Style\NumberFormat;

class AuthenticationStatsExport extends AbstractStatsExport implements WithColumnFormatting, WithStrictNullComparison
{
    protected StatisticsBetaRepository $repository;

    public function __construct(
    protected array $filters
  ) {
        parent::__construct('authentication');
        $this->repository = new StatisticsBetaRepository();
    }

    public function querySize(): int
    {
        return $this->query()->get()->count();
    }

    public function query(): Builder
    {
        return $this->repository->getAuthenticationStatsQuery($this->filters);
    }

    public function map($row): array
    {
        $user = User::find($row->user_uuid);

        $mapped = [
            $user->full_name,
            $user->email,
            $user->roles
                ->filter(fn ($role) => ! Str::startsWith($role->name, ['workflow-', 'group-']))
                ->map(fn ($role) => Role::getHumanName($role->name))
                ->implode(','),
        ];

        $mapped = array_merge($mapped, parent::map($user));

        $neverAuthenticated = $this->repository->neverAuthenticatedOnlyFiltered($this->filters);

        if (! $neverAuthenticated) {
            $mapped[] = Date::dateTimeToExcel($user->last_login_at);
            $mapped[] = $row->connections;
        }

        return $mapped;
    }

    public function headings(): array
    {
        $headings = [
            Lang::get('statistics.authentication.headings.fullname'),
            Lang::get('statistics.authentication.headings.email'),
            Lang::get('statistics.authentication.headings.roles'),
        ];
        $headings = array_merge($headings, parent::headings());
        $neverAuthenticated = $this->repository->neverAuthenticatedOnlyFiltered($this->filters);
        if (! $neverAuthenticated) {
            $headings[] = Lang::get('statistics.authentication.headings.last_login');
            $headings[] = Lang::get('statistics.authentication.headings.connection_count');
        }

        return $headings;
    }

    public function columnFormats(): array
    {
        $neverAuthenticated = $this->repository->neverAuthenticatedOnlyFiltered($this->filters);
        if (! $neverAuthenticated) {
            $lastConnexionColumn = chr(ord('D') + count(config('statistics.export.'.$this->type.'.headers', [])));

            return [
                $lastConnexionColumn => NumberFormat::FORMAT_DATE_DDMMYYYY,
            ];
        }

        return [];
    }

    public function getCsvSettings(): array
    {
        return [
            'delimiter' => ';',
            'enclosure' => '"',
            'line_ending' => PHP_EOL,
            'use_bom' => true,
            'include_separator_line' => false,
            'excel_compatibility' => false,
        ];
    }
}
