<?php

namespace Inside\Statistics\Console;

use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Inside\Console\Command;
use Inside\Statistics\Models\AuthenticationStatistic as OldAuthenticationStatistics;
use Inside\Statistics\Models\AuthenticationStatistics;
use Inside\Statistics\Models\Statistic as OldStatistics;

class MigrateStatisticsCommand extends Command
{
    /**
     * @var string
     */
    protected $signature = 'inside:stats:migrate {--force}';

    /**
     * @var string
     */
    protected $description = 'Migrate old stats data to new statistics tables.';

    /**
     * @return void
     */
    public function handle(): void
    {
        if (! $this->option('force') && ! $this->confirm('The command must be executed once. Are you sure to continue with the execution? [It is recommended to download the database backup before execution.]')) {
            $this->info('Execution aborted !');
            exit;
        }

        $this->info('Migrate Content stats ...');
        OldStatistics::select('*')->where('type', 'view')->groupBy('statisticable_type')->get()->pluck('statisticable_type')->each(function ($type) {
            $statsTableName = type_to_stats_table(class_to_type($type));
            if (! Schema::hasTable($statsTableName)) {
                $this->info("[Migrate-Statistics] Table '{$statsTableName}' does not exist.");
            }
        });

        OldStatistics::select('*')->where('type', 'view')->get()->each(function ($oldStat) {
            $statsTableName = type_to_stats_table(class_to_type($oldStat->statisticable_type));
            try {
                DB::table($statsTableName)->insert([
                    'user_uuid' => $oldStat->user_uuid,
                    'content_uuid' => $oldStat->statisticable_uuid,
                    'date' => get_date($oldStat->created_at)?->format('Y-m-d'),
                ]);
            } catch (\Exception $e) {
                // old content has been deleted
            }
        });
        $this->info('Migrate Content stats have been terminated!');

        $this->info('Migrate authentication stats ...');
        $oldAuthenticationStatistics = OldAuthenticationStatistics::select([
            'date',
            'uuid',
            DB::raw('SUM(connection_count) as total_connections'),
        ])->groupBy('date', 'uuid')->get();
        $oldAuthenticationStatistics->each(function ($statistic) {
            try {
                AuthenticationStatistics::insert([
                    'user_uuid' => $statistic->uuid,
                    'created_at' => get_date($statistic->date)?->format('Y-m-d'),
                    'connections' => $statistic->total_connections,
                ]);
            } catch (\Exception $e) {
                $this->info('[Migrate-Statistics] Exception: '.$e->getMessage());
            }
        });
        $this->info('Migrate authentication stats have been terminated!');
    }
}
