<?php

namespace Inside\Settings\Http\Controllers;

use Illuminate\Contracts\Filesystem\FileNotFoundException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;
use Inside\Permission\Exceptions\AuthorizationException;
use Inside\Settings\Contracts\SettingService;
use Inside\Settings\Facades\Settings;
use Inside\Settings\Models\Setting;
use Inside\Support\Str;
use InvalidArgumentException;
use Laravel\Lumen\Routing\Controller;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use ZipArchive;

/**
 * Class SettingController
 *
 * @OA\Tag(name="Paramètres interface",description="
# Introduction

Il s'agit des services déstinés à la gestion des paramètres de l'interface graphique.

")
 */
class SettingController extends Controller
{
    /**
     * Currently stored settings
     *
     *
     * @OA\Schema(
     *           schema="Setting",
     *           description="Un paramètre d'interface",
     *           type="object",
     *           @OA\Property(
     *               type="string",
     *               property="group",
     *               description="Le groupe/famille auquel appartient le paramètre",
     *               example="my-group"
     *           ),
     *           @OA\Property(
     *               type="string",
     *               property="key",
     *               description="La clé du paramètre",
     *               example="parameter1"
     *           ),
     *           @OA\Property(
     *               type="string",
     *               property="value",
     *               description="La valeur du paramètre",
     *               example="My value"
     *           )
     *   )
     *
     * @OA\Schema(
     *           schema="Settings",
     *           description="La liste des règlages d'un groupe/famille.",
     *           type="object",
     *           @OA\AdditionalProperties(ref="#/components/schemas/Setting"),
     *           example={"parameter1":"value1","parameter2":"value2","parameter3":"value3"}
     *   )
     *
     * @OA\Schema(
     *           schema="GlobalSettings",
     *           description="La liste des règles par groupe/famille de règlages",
     *           type="object",
     *           @OA\AdditionalProperties(ref="#/components/schemas/Settings"),
     *           example={"group1":{"parameter1":"value1","parameter2":"value2","parameter3":"value3"},"group2":{"parameter4":"value4","parameter5":"value5","parameter6":"value6"}}
     *   )
     *
     *
     * @OA\Get(
     *      path="/settings/{group}",
     *      operationId="SettingsList",
     *      tags={"Paramètres interface"},
     *      summary="Permet de récupérer la liste des paramètres d'interface par groupe/famille. Le retour Setting ne
     *      renseigne pas le group qui est implicite.",
     *     @OA\Parameter(
     *        name="group",
     *        in="path",
     *        example="my-group",
     *        required=false,
     *        @OA\Schema(type="string"),
     *        description="Indique le groupe/famille désiré. Le nom du groupe/famille est un nom type **machine**, il
    est donc composé d'alphanumérique sans caractères spéciaux ni espace ( genre slug )."
     *     ),
     * @OA\Response(
     *          response=200,
     *          description="Les paramètres",
     *          @OA\JsonContent(ref="#/components/schemas/Settings"),
     *      @OA\Response(response=400, description="Bad request"),
     *      @OA\Response(response=401, description="Not allowed")
     *    )
     * )
     *
     * @OA\Get(
     *      path="/settings",
     *      operationId="GlobalSettingsList",
     *      tags={"Paramètres interface"},
     *      summary="Permet de récupérer tous les paramètres stockés. Ils sont retournés indexés par famille.
    ",
     *     @OA\Response(
     *          response=200,
     *          description="Les paramètres",
     *          @OA\JsonContent(ref="#/components/schemas/GlobalSettings")
     *          ),
     *      @OA\Response(response=400, description="Bad request"),
     *      @OA\Response(response=401, description="Not allowed")
     * )
     *
     * @param  SettingService  $service
     * @param  string|null  $group
     * @return JsonResponse
     */
    public function listSettings(SettingService $service, ?string $group = null): JsonResponse
    {
        // Force $slug to be a correct slug on group name
        if ($group) {
            $group = Str::slug($group);
        }

        return new JsonResponse($service->listForGroup($group, is_maecia_admin()));
    }

    /**
     * @param  SettingService  $service
     * @param  string  $key
     * @return JsonResponse
     */
    public function listOptions(SettingService $service, string $key): JsonResponse
    {
        return new JsonResponse([
            $key => $service->get('#options', $key),
        ]);
    }

    /**
     * Store Settings
     *
     * @OA\Put(
     *      path="/settings",
     *      operationId="GlobalSettingsStore",
     *      tags={"Paramètres interface"},
     *      summary="Permet de stocker tous les paramètres d'interface. Si le paramètre n'existe pas, il sera alors
     *      créé.",
     *     @OA\RequestBody(
     *        description="Les paramètre d'interface à **stocker**",
     *        required=true,
     *        @OA\JsonContent(
     *             type="array",
     *             @OA\Items(ref="#/components/schemas/Setting"),
     *           )),
     *      @OA\Response(
     *          response=200,
     *          description="Indique que l'opération est un succès"
     *      ),
     *      @OA\Response(response=400, description="Bad request"),
     *      @OA\Response(response=401, description="Not allowed")
     * )
     */
    public function storeSetting(Request $request, SettingService $service): JsonResponse
    {
        foreach ($request->all() as $setting) {
            $validator = Validator::make(
                $setting,
                [
                    'group' => 'required|alpha_dash|max:255',
                    'key'   => 'required|alpha_dash|max:255',
                    'value' => 'nullable',
                ]
            );

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            if (Str::startsWith('_', $setting['group']) && ! is_maecia_admin()) {
                throw new AuthorizationException();
            }

            if ($service->isCurrentValueMasked($setting['group'], $setting['key'], $setting['value'])) {
                continue;
            }

            Setting::updateOrCreate(
                [
                    'group' => $setting['group'],
                    'key'   => $setting['key'],
                ],
                [
                    'value' => $setting['value'],
                ]
            );
        }

        return new JsonResponse(['successful' => true]);
    }

    public function storeOption(Request $request): JsonResponse
    {
        foreach ($request->all() as $setting) {
            $validator = Validator::make(
                $setting,
                [
                    'key'   => 'required|alpha_dash|max:255',
                    'value' => 'nullable',
                ]
            );
            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            Setting::updateOrCreate(
                [
                    'group' => '#options',
                    'key'   => $setting['key'],
                ],
                [
                    'value' => $setting['value'],
                ]
            );
        }

        return new JsonResponse(['successful' => true]);
    }

    /**
     * Export all designs settings in a zip file
     *
     * @return BinaryFileResponse
     * @throws FileNotFoundException
     */
    public function exportDesigns()
    {
        // Export layouts as a big json data
        $data = Settings::getExportableSettings()->toJson(JSON_PRETTY_PRINT);

        // Prepare a zip
        $zipFilenameParts = [
            'designs',
            config('app.code'),
            inside_version(),
            Carbon::now()->timestamp,
            Str::random(64),
        ];
        $zipFilename = implode('_', $zipFilenameParts).'.zip';
        $zipPath = str_replace('/', DIRECTORY_SEPARATOR, storage_path('app/tmp/'.$zipFilename));
        $zipFile = new ZipArchive();
        $zipFile->open($zipPath, ZipArchive::CREATE);
        $zipFile->addFromString('designs.json', $data);
        $zipFile->close();

        // Download zip
        $response = response()->download(
            $zipPath,
            $zipFilename,
            ['Content-Type: application/octet-stream', 'Content-Length: '.filesize($zipPath)]
        );

        if (! windows_os()) {
            $response->deleteFileAfterSend(true);
        }

        return $response;
    }

    public function importDesigns(Request $request): JsonResponse
    {
        // Get archive
        $zipPath = $request->get('designs');
        if (! $zipPath) {
            throw new InvalidArgumentException('Missing designs file path');
        }
        if (! Storage::disk('local')->exists($zipPath)) {
            throw new InvalidArgumentException('File path does not exist');
        }

        $zipPath = Storage::disk('local')->path($zipPath);
        $zipFile = new ZipArchive();
        $zipFile->open($zipPath, ZipArchive::CREATE);
        $data = $zipFile->getFromName('designs.json');
        $zipFile->close();
        if (! $data) {
            throw new InvalidArgumentException('Wrong designs file');
        }
        $data = json_decode($data);
        if (! $data) {
            throw new InvalidArgumentException('Wrong designs file');
        }
        $data = icollect($data);

        $success = Settings::importSettings($data);

        if ($success) {
            Artisan::call('cache:clear');
        }

        return new JsonResponse([
            'success' => $success,
        ]);
    }
}
