<?php

namespace Inside\Settings\Contracts;

use Illuminate\Contracts\Filesystem\FileNotFoundException;
use Illuminate\Support\Collection;

interface SettingService
{
    /**
     * Get all settings
     *
     * @return array
     */
    public function all(): array;

    /**
     * @param  string|null  $group
     * @param  bool  $isMaeciaAdmin
     * @return Collection
     */
    public function listForGroup(?string $group = null, bool $isMaeciaAdmin = false): Collection;

    /**
     * Does $group/$key setting exists ?
     *
     * @param  string  $group
     * @param  string  $key
     * @return bool
     */
    public function has(string $group, string $key): bool;

    /**
     * Get a setting by its group & key, getting $default if setting does not exists
     *
     * @param  string  $group
     * @param  string  $key
     * @param  string|null  $default
     * @return string|null
     */
    public function get(string $group, string $key, ?string $default = null): ?string;

    /**
     * Set a $group/$key setting
     *
     * @param  string  $group
     * @param  string|array  $key
     * @param  string|null  $value
     * @return self
     */
    public function set(string $group, $key, ?string $value = null): self;

    /**
     * Forget a $group/$key setting
     *
     * @param  string  $group
     * @param  string  $key
     */
    public function forget(string $group, string $key): self;

    /**
     * Forget all settings
     *
     * @return $this
     */
    public function forgetAll(): self;

    /**
     * get a collection of exportable settings
     *
     * @return Collection
     * @throws FileNotFoundException
     */
    public function getExportableSettings(): Collection;

    /**
     * Import settings from a $data collection of settings
     *
     * @param  Collection  $data
     * @return bool
     */
    public function importSettings(Collection $data): bool;

    /**
     * Check if the current value of a setting is masked
     *
     * @param  string  $group
     * @param  string  $key
     * @param  string  $value
     * @return bool
     */
    public function isCurrentValueMasked(string $group, string $key, ?string $value): bool;
}
