<?php

namespace Inside\Settings\Audit\Tests;

use Inside\Settings\Audit\Interfaces\TestInterface;

abstract class BaseTest implements TestInterface
{
    private static array $messages = [];

    public const STATUS_SUCCESS = 1;

    public const STATUS_PARTIAL = 2;

    public const STATUS_FAILED = 3;

    public const STATUS_IGNORED = 4;

    protected static array $childTests = [];

    public static function getLabel(): string
    {
        return 'There is no label defined !';
    }

    public static function getMessage(): string
    {
        return static::$messages[static::class] ?? '';
    }

    protected static function setMessage(string $message): void
    {
        static::$messages[static::class] = $message;
    }

    public static function execute(): array
    {
        if (static::shouldIgnore()) {
            return static::formatResult(self::STATUS_IGNORED, static::getMessage());
        }

        if (! empty(static::$childTests)) {
            return static::executeChildTests();
        }

        return static::formatResult(static::check() ? self::STATUS_SUCCESS : self::STATUS_FAILED, static::getMessage());
    }

    public static function check(): bool
    {
        if (! empty(static::$childTests)) {
            throw new \LogicException(static::class.' has child tests and should not implement check().');
        }

        throw new \LogicException(static::class.' must implement check() method.');
    }

    public static function formatResult(int $status, string $message = '', array $results = []): array
    {
        return [
            'label' => static::getLabel(),
            'status' => $status,
            'status_symbol' => static::getStatusSymbol($status),
            'message' => $message,
            'results' => $results,
        ];
    }

    public static function shouldIgnore(): bool
    {
        return false;
    }

    private static function executeChildTests(): array
    {
        $results = [];
        $statuses = [];

        foreach (static::$childTests as $childTest) {
            if ($childTest::shouldIgnore()) {
                $results[] = $childTest::formatResult($childTest::STATUS_IGNORED);
                continue;
            }

            $childResult = $childTest::execute();
            $results[] = $childResult;
            $statuses[] = $childResult['status'];
        }

        return static::formatResult(static::calculateFinalStatus($statuses), '', $results);
    }

    private static function calculateFinalStatus(array $statuses): int
    {
        if (empty($statuses)) {
            return self::STATUS_IGNORED;
        }

        if (count(array_filter($statuses, fn ($status) => $status === self::STATUS_SUCCESS)) === count($statuses)) {
            return self::STATUS_SUCCESS;
        }

        if (count(array_filter($statuses, fn ($status) => $status === self::STATUS_FAILED)) === count($statuses)) {
            return self::STATUS_FAILED;
        }

        return self::STATUS_PARTIAL;
    }

    protected static function getStatusSymbol(int $status): string
    {
        return match ($status) {
            1 => '🟢',
            2 => '🟠',
            3 => '🔴',
            4 => '⚫️',
            default => '❓',
        };
    }
}
