<?php

namespace Inside\Search\Http\Controllers;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Inside\Content\Facades\Schema;
use Inside\Permission\Facades\Permission;
use OpenApi\Annotations as OA;

class SearchableIndex extends Controller
{
    public function __invoke(Request $request): JsonResponse
    {
        $langcode = $request->get('langcode');

        $searchable = [];
        foreach (Schema::getContentTypes() as $type) {
            $modelOptions = Schema::getModelOptions($type);
            if (! array_key_exists('searchable', $modelOptions)
                || ! $modelOptions['searchable']
            ) {
                continue;
            }

            if (Permission::allowed('read', $type)) {
                $searchable[] = [
                    'type'   => $type,
                    'title'  => (isset($langcode) && array_key_exists($langcode, $modelOptions['title']))
                        ? $modelOptions['title'][$langcode] : $modelOptions['title'],
                    'global' => (bool) ($modelOptions['global_searchable'] ?? false),
                ];
            }
        }

        return response()->json($searchable);
    }
}

/**
 * @OA\Schema(
 *     schema="SearchableResult",
 *     type="array",
 *          items=@OA\Items(type="object",
 *             @OA\Property(
 *                         property="type",
 *                        description="Le type ( nom machine ) du contenu recherchable.",
 *                        type="string",
 *                        example="news"
 *                      ),
 *                      @OA\Property(
 *                         property="global",
 *                        description="Ce type est il utilisé pour la recherche globale.",
 *                        type="boolean",
 *                        example=false
 *                      ),
 *                       @OA\Property(
 *                         property="title",
 *                        description="Le titre dans toutes les langues. Une propriété par langue, le nom de
 * la propriété est l'identifiant de langue ( exemple: français => fr, anglais => en, espagnol => es ) ...",
 *                        type="object",
 *                         @OA\Property(
 *                         property="fr",
 *                        description="Le titre en français.",
 *                        type="string",
 *                        example="Actualités"
 *                      ), @OA\Property(
 *                         property="en",
 *                        description="Le titre en anglais.",
 *                        type="string",
 *                        example="News"
 *                      ),
 *                 )
 *             )
 *     )
 *
 * @OA\Get(
 *      path="/searchable",
 *      operationId="searchable",
 *      tags={"Recherche"},
 *      summary="Indique tous les types de contenu où l'on peut effectuer une recherche ainsi que le titre dans
 *  toutes les langues pour chaque type.",
 *      @OA\Response(
 *          response=200,
 *          description="Les types recherchables",
 *          @OA\JsonContent(ref="#/components/schemas/SearchableResult")
 *       ),
 *      @OA\Response(response=400, description="Bad request")
 *     )
 */
