<?php

namespace Inside\Search\Helpers;

use Exception;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;
use function in_array;
use Inside\Content\Models\Model;

/**
 * Search controller
 *
 * @category Class
 * @author   Maecia <contact@feldoe.net>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class Searchable
{
    public const OPERATOR_OR = 0;

    public const OPERATOR_AND = 1;

    protected ?Collection $classes = null;

    protected ?Collection $globalClasses = null;

    protected array $knownTraits = [
        \Inside\Search\Solr\Models\Traits\Searchable::class,
        \Inside\Search\Database\Models\Traits\Searchable::class,
    ];

    /**
     * Get All searchable classes
     *
     * @return Collection
     * @throws Exception
     */
    public function getSearchableClasses(): Collection
    {
        $appKey = env('APP_KEY');

        if (Cache::has('inside.'.$appKey.'.search.classes')) {
            $this->classes = Cache::get('inside.'.$appKey.'.search.classes');

            return $this->classes;
        }
        $this->classes = $this->listSearchableModels();
        // \Cache::forever('inside.' . $appKey . '.search.classes', $this->classes);

        return $this->classes;
    }

    /**
     * Trying to guess our searchable classes
     *
     * @throws Exception
     */
    public function listSearchableModels(): Collection
    {
        $models = Model::all();
        $dynamicModelClasses = collect();

        foreach ($models as $model) {
            if (isset($model->options['searchable']) && $model->options['searchable']) {
                $class = (string) str_replace('Inside\\Content\\Models\\Contents\\', '', $model->model);
                $class = (string) str_replace('Inside\\Content\\Models\\Sections\\', '', $class);

                $dynamicModelClasses[snake_case($class)] = $model;
            }
        }

        return $dynamicModelClasses;
    }

    public function isClassSearchable(string $class): bool
    {
        if (! in_array(\Laravel\Scout\Searchable::class, class_uses_recursive($class), true)) {
            return false;
        }

        return true;
    }

    /**
     * Get All global searchable classes
     *
     * @throws Exception
     */
    public function getGlobalSearchableClasses(): Collection
    {
        $appKey = env('APP_KEY');

        if (Cache::has('inside.'.$appKey.'.global.search.classes')) {
            $this->globalClasses = Cache::get('inside.'.$appKey.'.global.search.classes');

            return $this->globalClasses;
        }

        $this->globalClasses = $this->listGlobalSearchableModels();

        return $this->globalClasses;
    }

    /**
     * Trying to guess our searchable classes
     *
     * @throws Exception
     */
    public function listGlobalSearchableModels(): Collection
    {
        $models = Model::all();
        $dynamicModelClasses = collect();

        foreach ($models as $model) {
            if (isset($model->options['global_searchable']) && $model->options['global_searchable']) {
                $class = str_replace('Inside\\Content\\Models\\Contents\\', '', $model->model);
                $class = str_replace('Inside\\Content\\Models\\Sections\\', '', $class);

                $dynamicModelClasses[class_to_type($model->model)] = $model;
            }
        }

        return $dynamicModelClasses;
    }
}
