<?php

namespace Inside\Search\Console;

use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Str;
use Inside;
use Inside\Facades\Package;
use Inside\Search\Facades\Searchable;

/**
 * Class IndexCommand
 *
 * @category Class
 * @author   Maecia <contact@feldoe.net>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class IndexCommand extends Command
{
    /**
     * @var string
     */
    protected $name = 'index:rebuild';

    /**
     * @var string
     */
    protected $signature = 'index:rebuild
                            {--S|silent=false}
                            {--T|type= }';

    /**
     * @var string
     */
    protected $description = 'Reindex all searchable models';

    /**
     * Handle index command
     *
     * @return void
     * @throws \Exception
     */
    public function handle(): void
    {
        /** @var string|null $silent */
        $silent = $this->option('silent');
        $silent = $silent === null;
        $now = Carbon::now();
        $searchables = Searchable::getSearchableClasses();

        // Check version
        if (Package::has('inside-search-database') && ! Schema::hasColumn('inside_search_database_indexes', 'title')) {
            return; // Old version must wait for search migration to be played
        }

        $silent or $this->info(Inside::getCommandInfo());

        if ($type = $this->option('type')) {
            /** @var string $type */
            $silent or $this->info('Only reindex type ['.$type.']');
        }

        $silent or $this->info('Deleting old index');

        // Check type
        if ($type) {
            $allowedTypes = $searchables->filter(
                function ($definition) {
                    return Str::startsWith($definition->model, 'Inside\\Content\\Models\\Contents\\');
                }
            )->keys();

            if (! $allowedTypes->contains($type)) {
                $silent or $this->error(
                    'Type ['.$type."] is not known.\n Known types:\n".$allowedTypes->implode("\n")
                );
                exit(-1);
            }
        }

        if ($type) {
            // Delete type entities
            $class = $searchables[$type];

            if (config('scout.driver', 'mysql') != 'solr') {
                $query = call_user_func(search_namespace('Models\Index::query'));
                $query->where('indexable_type', $class->model)->delete();
            }
        } else {
            // Truncate index table
            call_user_func(search_namespace('Models\Index::truncate'));
        }

        $silent or $this->info('DONE');
        $silent or $this->line('');

        $silent or $this->info(
            'Starting to index at ['.$now->toFormattedDateString().' '.$now->toTimeString().']'
        );

        if ($type) {
            $class = $searchables[$type];

            $query = call_user_func($class->model.'::query');

            if ($query) {
                $total = $query->count();
                $i = 0;
                while ($i < $total) {
                    $items = $query->skip($i)->take(config('scout.chunk.searchable'));
                    if ($items) {
                        $i += $items->count();
                        $items->searchable();
                    }
                }
            }
        } else {
            inside_search_force_reindex(! $silent ? $this : null);
        }

        $silent or $this->line('');
        $silent or $this->info('Indexation done, started ['.$now->diffForHumans().']');
    }
}
