<?php

namespace Inside\Reaction\Validators;

use Illuminate\Support\Facades\Validator as ValidatorFacade;
use Illuminate\Validation\Validator;
use Inside\Reaction\Exceptions\ReactionValidatorException;
use Inside\Reaction\Exceptions\UniqueReactionException;
use Inside\Reaction\Facades\Reactions;

class ReactionValidator
{
    /**
     * @var array
     */
    protected $rules = [
        'user_uuid' => 'required|string|size:36',
        'reactionable_uuid' => 'required|string|size:36',
        'reactionable_type' => 'required|string',
        'type' => 'required|string',
    ];

    /**
     * @param mixed $data
     * @return void
     * @throws ReactionValidatorException
     * @throws UniqueReactionException
     */
    public function validate($data): void
    {
        /** @var Validator $validator */
        $validator = ValidatorFacade::make($data, $this->rules);
        if ($validator->fails()) {
            $errors = $this->formatErrors($validator);
            throw new ReactionValidatorException(implode(' ', $errors), 400);
        }

        $langcode = $data['langcode'] ?? null;
        $reactions = Reactions::get(
            $data['type'],
            $data['reactionable_uuid'],
            $data['user_uuid'],
            $langcode,
            $data['reactionable_type'],
            ['uuid'],
            false
        );

        if (count($reactions)) {
            throw new UniqueReactionException('User already reacted', 400);
        }
    }

    protected function formatErrors(Validator $validator): array
    {
        $errors = [];
        $messages = $validator->getMessageBag();

        foreach ($messages->keys() as $key) {
            $errors[$key] = implode('\\n', $messages->get($key));
        }

        return $errors;
    }
}
