<?php

declare(strict_types=1);

namespace Inside\Permission\Services;

use Closure;
use Exception;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Database\Query\Builder;
use Illuminate\Database\QueryException;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Str;
use Inside\Authentication\Models\ApplicationClient;
use Inside\Authentication\Models\User;
use Inside\Content\Contracts\SchemaService;
use Inside\Content\Exceptions\FieldSchemaNotFoundException;
use Inside\Content\Facades\Schema as InsideSchema;
use Inside\Content\Models\Content;
use Inside\Content\Models\Contents\Users;
use Inside\Facades\Inside;
use Inside\Host\Exodus\Services\ContentTypeStatusService;
use Inside\Permission\Events\PermissionCreatedEvent;
use Inside\Permission\Events\PermissionDeletedEvent;
use Inside\Permission\Exodus\Actions\PermissionLogic\User\CanCreateContent;
use Inside\Permission\Exodus\Actions\PermissionLogic\User\CanDeleteContent;
use Inside\Permission\Exodus\Actions\PermissionLogic\User\CanReadContent;
use Inside\Permission\Exodus\Actions\PermissionLogic\User\CanUpdateContent;
use Inside\Permission\Exodus\Dto\Privileges\BackofficePrivilegeDto;
use Inside\Permission\Exodus\Enums\BackofficeEnum;
use Inside\Permission\Exodus\Services\RolePrivilegesService;
use Inside\Permission\Facades\Role as RoleService;
use Inside\Permission\Models\Permission;
use Inside\Permission\Models\Role;
use InvalidArgumentException;

/**
 * Service handle any permission
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
final class PermissionService
{
    /**
     * List of permissible models
     */
    protected array $permissibleClasses = [];

    /**
     * List of categorizable models
     */
    protected array $categorizableClasses = [];

    /**
     * Avoid infinite loop
     */
    protected array $tempStore = [];

    /**
     * Saved permissions
     */
    protected array $permissions = [];

    /**
     * role Ids for a user
     */
    protected array $roleIds = [];

    /**
     * is Admin by user uuids
     */
    protected array $admins = [];

    /**
     * stored users permission by uuid
     */
    protected array $users = [];

    /**
     * Current user
     */
    protected User|null $user = null;

    /**
     * Does allowedscope is enable ?
     */
    protected bool $allowedScopeEnable = true;

    /**
     * backoffice Entries memory cache
     */
    protected ?Collection $backofficeEntriesAccessCache = null;

    /**
     * Initiate service
     */
    public function __construct(
        protected SchemaService $schemaService
    ) {
        if (static::isSystemV2Enabled()) {
            $this->permissibleClasses = app(ContentTypeStatusService::class)->permissibles()->keys()->map(fn (string $type) => type_to_class($type))->toArray();
            $this->categorizableClasses = app(ContentTypeStatusService::class)->categorizables()->keys()->map(fn (string $type) => type_to_class($type))->toArray();
        } else {
            foreach (InsideSchema::getContentTypes() as $type) {
                $modelOptions = InsideSchema::getModelOptions($type);
                if (isset($modelOptions['permissible']) && $modelOptions['permissible']) {
                    $this->permissibleClasses[] = type_to_class($type);
                }
                if (isset($modelOptions['categorizable']) && $modelOptions['categorizable']) {
                    $this->categorizableClasses[] = type_to_class($type);
                }
            }
            $this->loadBackofficeEntriesAccessCache();
        }
    }

    public static function isSystemV2Enabled(): bool
    {
        return (bool) setting('permission', 'system_v2_enabled', false);
    }

    public static function isSystemV2Migrated(): bool
    {
        return (bool) setting('permission', 'system_v2_migrated', false);
    }

    public static function isSystemV2Initialized(): bool
    {
        return (bool) setting('permission', 'system_v2_initialized', false);
    }

    /**
     * Does class is permissible ?
     */
    public function isPermissible(string $class): bool
    {
        return array_search($class, $this->permissibleClasses) !== false;
    }

    protected function getRoleAsFieldAllowedValues(string $locale, bool $withCount = false): array
    {
        $roles = RoleService::tree();
        $allowedValues = [];
        foreach ($roles as $role) {
            $allowedValue = [
                'title' => RoleService::getHumanName($role['name'], $locale),
                'uuid' => $role['id'] + ($role['type'] === 'role' ? 0 : 10000), // avoid id collision
                'type' => $role['type'],
                'children' => [],
            ];
            if ($role['type'] === 'role' && $withCount) {
                $allowedValue['qualifiers'] = [
                    [
                        'title' => RoleService::getUserCountForRole($role['id']),
                        'color' => setting('_theme', 'colorPrimary'),
                        'position' => 'after',
                    ],
                ];
            }

            foreach ($role['children'] as $roleChildren) {
                $roleInformation = [
                    'title' => RoleService::getHumanName($roleChildren['name'], $locale),
                    // avoid id collision
                    'uuid' => $roleChildren['id'] + ($roleChildren['type'] === 'role' ? 0 : 10000),
                    'type' => $roleChildren['type'],
                ];
                if ($roleChildren['type'] === 'role' && $withCount) {
                    $roleInformation['qualifiers'] = [
                        [
                            'title' => RoleService::getUserCountForRole($roleChildren['id']),
                            'color' => setting('_theme', 'colorPrimary'),
                            'position' => 'after',
                        ],
                    ];
                }
                $allowedValue['children'][] = $roleInformation;
            }
            $allowedValues[] = $allowedValue;
        }

        return $allowedValues;
    }

    /**
     * get role select picker to be sent as a form field to the front
     */
    public function getRolePickerFieldForFrontForm(
        string $fieldName,
        string $translationKey = 'roles.label',
        ?Content $content = null,
        array $defaultValue = [],
        int $weight = 0,
        bool $withCount = false,
        array $rolesDescriptions = []
    ): array {
        $rolesLabels = [];
        $rolesAllowedValues = [];
        foreach (list_languages() as $locale) {
            $rolesLabels[$locale] = trans($translationKey, [], $locale);
            $rolesAllowedValues[$locale] = $this->getRoleAsFieldAllowedValues($locale, $withCount);

            if ($fieldName === 'notifications_roles') {
                $rolesDescriptions[$locale] = trans('notifications.forms.title.roles.description', [], $locale);
            }
        }

        $field = [
            'name' => $fieldName,
            'options' => [
                'cardinality' => -1,
                'categorizable' => false,
                'allowed_values' => [
                    'roles' => $rolesAllowedValues,
                ],
                'default' => $defaultValue,
                'title' => $rolesLabels,
                'description' => $rolesDescriptions,
                'widget' => 'selectroles',
            ],
            'type' => 'select',
            'weight' => $weight,
        ];

        if ($fieldName === 'newsletter_roles') {
            $field['options']['required'] = true;
        }

        if (! is_null($content)) {
            if (! InsideSchema::hasField(class_to_type($content), $fieldName)) {
                throw FieldSchemaNotFoundException::named(get_class($content), $fieldName);
            }

            $roleIds = $content->{$fieldName} ? json_decode($content->{$fieldName}) : [];
            if (is_array($roleIds)) {
                $field['value'] = $roleIds;
            }
        }

        return $field;
    }

    /**
     * Are we allowed to do this
     *
     * Easier check than can
     */
    public function allowed(
        string $action,
        string $type,
        string $uuid = null,
        User $user = null
    ): bool {
        return (static::isSystemV2Enabled())
            ? $this->allowedV2($action, $type, $uuid, $user)
            : $this->allowedV1($action, $type, $uuid, $user);
    }

    protected function allowedV1(
        string $action,
        string $type,
        string $uuid = null,
        User $user = null
    ): bool {
        $me = $user ?? $this->user();
        if (! $me) {
            return false;
        }
        if ($me->isSuperAdmin()) {
            return true;
        }

        if (in_array($action, ['delete', 'update']) && $uuid == null) {
            return false;
        }

        if ($type === 'users' && $action === 'create' && $uuid !== null) {
            return true; // Special exception on users ( populate users selects )
        }

        // Special user case ( users is non permissible, only super admin can
        // create / update / delete users ) but we all as well users that
        // has access to backoffice user module !
        if (
            $type === 'users' && in_array(
                $action,
                ['create', 'update', 'delete']
            ) && Gate::forUser($me)->allows('access_user')
        ) {
            return true;
        }

        $class = type_to_class($type);

        // Non-permissible contents can always be read
        if (! $this->isPermissible(type_to_class($type))) {
            if ($type === 'comments' && $action === 'delete') {
                return call_user_func($class.'::find', $uuid)?->author === $me->uuid;
            }

            return ($action === 'read') || ($type === 'users' && $uuid === $me->uuid);
        }

        if (
            ($action == 'create' && $uuid === null)
            || ($action == 'read'
                && $uuid === null)
        ) {
            // Note: on peut avoir une demande 'create' avec un uuid dans le cas ou on veut qualifier un select.
            $model = new $class();
        } else {
            $model = call_user_func($class.'::find', $uuid);
            if (! $model) {
                return false;
            }
        }
        if ($this->can($action, $model, $me)) {
            return true;
        }

        // Let's try with one of our callbacks
        $customPermissions = config('permission.custom_permissions', []);

        foreach ($customPermissions as $customPermission) {
            if ($customPermission && is_callable($customPermission)) {
                if ($customPermission($action, $type, $uuid, $me)) {
                    return true;
                }
            }
        }

        // No luck!
        return false;
    }

    protected function allowedV2(
        string $action,
        string $type,
        string $uuid = null,
        User $user = null
    ): bool {
        $me = $user ?? $this->user();

        if (! $me) {
            return false;
        }

        if (! InsideSchema::isContentType($type) && $action === 'read') {
            return true;
        }

        /** @var Content $class */
        $class = type_to_class($type);

        $model = is_null($uuid) ? new $class() : call_user_func($class.'::find', $uuid);

        if (! $model) {
            return false;
        }

        return $this->canDoAction($action, $model, $me);
    }

    /**
     * Check if we can do this action on this model
     *
     * NOTE: this is direct permission check, bypass overrides, use ::allowed
     * if you want overrides instead
     */
    public function can(
        string $action = 'read',
        ?Content $model = null,
        ?User $user = null
    ): bool {
        return (static::isSystemV2Enabled())
            ? $this->canV2($action, $model, $user)
            : $this->canV1($action, $model, $user);
    }

    protected function canV1(
        string $action = 'read',
        ?Content $model = null,
        ?User $user = null
    ): bool {
        $permissionKey = $action.'-'.(is_null($model) ? 'all' : class_to_type($model)).'-'.($model->uuid ?? 'all').($user ? $user->uuid : '');
        if (! array_key_exists($permissionKey, $this->permissions)) {
            $this->permissions[$permissionKey] = $this->canDoAction(
                $action,
                $model,
                $user
            );
        }

        return $this->permissions[$permissionKey];
    }

    protected function canV2(
        string $action = 'read',
        ?Content $model = null,
        ?User $user = null
    ): bool {
        return $this->canDoAction($action, $model, $user);
    }

    protected function canDoAction(
        string $action = 'read',
        ?Content $model = null,
        ?User $user = null
    ): bool {
        return (static::isSystemV2Enabled())
            ? $this->canDoActionV2($action, $model, $user)
            : $this->canDoActionV1($action, $model, $user);
    }

    protected function canDoActionV1(
        string $action = 'read',
        ?Content $model = null,
        ?User $user = null
    ): bool {
        if (is_null($model)) {
            return false;
        }

        if (is_null($user)) {
            $user = $this->user();
        }

        if (is_null($user)) {
            return false;
        }

        if ($this->isSuperAdmin($user) || $model->uuid == $user->uuid) {
            return true;
        }

        if ($action == 'read' && ! in_array(get_class($model), $this->permissibleClasses)) {
            return true; // Non-permissible content can always been red
        }

        if (
            ! in_array(
                class_to_type($model),
                config('permission.non_implicit_permission_to_author_content_types', [])
            )
            && $model->exists
            && $user->uuid === $model->author
        ) {
            return true;
        }

        // Special user case
        if ($action == 'read' && class_to_type($model) == 'users') {
            return true; //
        }

        return DB::table('inside_permissions')
            ->when(
                $action == 'create' && ! $model->uuid,
                fn ($query) => $query->whereNull('inside_permissions.uuid')
            )
            ->where('inside_permissions.action', $action)
            ->where('inside_permissions.type', get_class($model))
            ->when($model->uuid, function ($query) use ($model) {
                $query->where(function ($query) use ($model) {
                    $query->where('inside_permissions.uuid', $model->uuid)
                        ->orWhereNull('inside_permissions.uuid'); // it means we can $action any uuids
                });
            })->where(function ($query) use ($user) {
                $query->where('inside_permissions.user_uuid', $user->uuid)
                    ->orWhereIn(
                        'inside_permissions.role_id',
                        $this->getRoleIdsForAUser($user)
                    );
            })->exists();
    }

    protected function canDoActionV2(
        string $action = 'read',
        ?Content $model = null,
        ?User $user = null
    ): bool {
        if (is_null($model)) {
            return false;
        }

        if (is_null($user)) {
            $user = $this->user();
        }

        if (is_null($user)) {
            return false;
        }

        // Check if a custom logic is defined for this case.
        // If one of the custom permission return true, we allow the action.
        // This is useful to add custom permission logic for specific cases like inside-groups.
        if (config('permission.custom_permissions')) {
            foreach (config('permission.custom_permissions') as $customPermission) {
                if ($customPermission($action, $model->content_type, $model->uuid, $user)) {
                    return true;
                }
            }
        }

        return match ($action) {
            'create' => (new CanCreateContent())->handle($user, $model),
            'update' => (new CanUpdateContent())->handle($user, $model),
            'delete' => (new CanDeleteContent())->handle($user, $model),
            default => (new CanReadContent())->handle($user, $model),
        };
    }

    /**
     * Get the User permission User
     */
    public function user(User|ApplicationClient|Users|null $user = null): ?User
    {
        if ($user instanceof ApplicationClient) {
            // JWT application client
            $user = User::where('email', config('app.technical_mail'))->first();
        }
        $key = $user !== null ? $user->uuid : '###ME###';
        if (! array_key_exists($key, $this->users)) {
            $this->users[$key] = $this->getUser($user);
        }

        return $this->users[$key];
    }

    /**
     * Reset user from memory
     */
    public function reset(?User $user): void
    {
        $key = $user != null ? $user->uuid : '###ME###';
        if (array_key_exists($key, $this->users)) {
            unset($this->users[$key]);
        }
    }

    public function withoutAllowedScope(Closure $callback): mixed
    {
        $this->disableAllowedScope();

        try {
            return $callback();
        } finally {
            $this->enableAllowedScope();
        }
    }

    public function isAllowedScopeEnable(): bool
    {
        return $this->allowedScopeEnable;
    }

    public function enableAllowedScope(): void
    {
        $this->allowedScopeEnable = true;
    }

    public function disableAllowedScope(): void
    {
        $this->allowedScopeEnable = false;
    }

    /**
     * get The user $user or current user
     */
    protected function getUser(User|Users|null $user = null): ?User
    {
        if ($user === null) {
            if (! $this->user && Auth::check()) {
                /** @var User $user */
                $user = Auth::user();
                $this->user = $user->load('roles');
            }

            return $this->user;
        }

        if ($user instanceof User) {
            return $user;
        } elseif ($user->uuid) {
            return User::with('roles')->find($user->uuid); // @phpstan-ignore-line
        }

        return null;
    }

    /**
     * is User a Super admin
     */
    public function isSuperAdmin(User $user): bool
    {
        if (! array_key_exists($user->uuid, $this->admins)) {
            $this->admins[$user->uuid] = $user->isSuperAdmin();
        }

        return isset($user->uuid) && $this->admins[$user->uuid];
    }

    /**
     * Get array of role ids for the user $user
     *
     * @param User $user
     *
     * @return array
     */
    public function getRoleIdsForAUser(User $user): array
    {
        if (! method_exists($user, 'roles')) {
            return [];
        }
        if (! array_key_exists($user->uuid, $this->roleIds)) {
            $this->roleIds[$user->uuid] = $user->roles()
                ->pluck('inside_roles.id')->toArray();
        }

        return $this->roleIds[$user->uuid];
    }

    /**
     * Build role for given role
     * @throws Exception
     */
    public function buildPermissionForRole(?int $roleId = null): void
    {
        if ($roleId == null) {
            $this->buildPermissionForAllRoles();
        } else {
            // Then generate permission for permissible things
            $this->generatePermissions($roleId);
        }
    }

    /**
     * Generate permission for all roles
     * @throws Exception
     */
    public function buildPermissionForAllRoles(): void
    {
        $roles = Role::where('pid', null)->get();

        foreach ($roles as $role) {
            $this->buildPermissionForRole($role->id);
        }
    }

    /**
     * Generate permission
     * @throws Exception
     */
    protected function generatePermissions(int $roleId): void
    {
        DB::beginTransaction();

        $this->deleteAllPermissions($roleId);
        $this->generateNonPermissibles($roleId);

        $schemas = DB::table('inside_permissions_schema')
            ->select(
                [
                    'invert',
                    'action',
                    'children',
                    'custom',
                    'authorizable_uuid',
                    'authorizable_type',
                ]
            )
            ->join(
                'inside_roles_permissions_schema',
                'inside_permissions_schema.id',
                'inside_roles_permissions_schema.permission_schema_id'
            )
            ->where('inside_roles_permissions_schema.role_id', $roleId)
            ->orderBy('is_content_specific')
            ->orderBy('authorizable_type')
            ->orderBy('children', 'desc')
            ->orderBy('invert')
            ->get();

        foreach ($schemas as $schema) {
            if ($schema->invert) {
                $this->deletePermission($roleId, (array) $schema);
            } else {
                $this->createPermission($roleId, (array) $schema);
            }
        }

        DB::commit();
    }

    /**
     * Delete permission for role_id
     * @throws Exception
     */
    protected function deleteAllPermissions(int $roleId): void
    {
        Permission::where('role_id', $roleId)->whereNotIn('type', ['custom', 'backoffice'])->delete();
    }

    /**
     * Generate non permissible for role (all content are readable)
     */
    protected function generateNonPermissibles(int $roleId): void
    {
        foreach (InsideSchema::getContentTypes() as $type) {
            $modelOptions = InsideSchema::getModelOptions($type);
            $permissions = [];

            if ($type != 'users' && isset($modelOptions['permissible']) && $modelOptions['permissible'] == 0) {
                $data = [
                    'action' => 'read',
                    'role_id' => $roleId,
                ];

                $table = type_to_table($type);

                if (! $table || ! Schema::hasTable($table)) {
                    continue;
                }

                $uuids = DB::table($table)->select('uuid')->get()->pluck('uuid')->toArray();
                foreach ($uuids as $uuid) {
                    $data['uuid'] = $uuid;
                    $data['type'] = type_to_class($type);
                    $permissions[] = $data;
                }

                try {
                    DB::table('inside_permissions')->insert($permissions);
                } catch (QueryException $exception) {
                    // Do nothing if duplicated content
                }
            }
        }
    }

    /**
     * Delete a permission
     */
    public function deletePermission(int $roleId, array $schema): void
    {
        if (env('INSIDE_DEBUGING_PERMISSIONS', false) == true) {
            Log::debug(__('deletePermission :roleId', ['roleId' => $roleId]), $schema);
        }
        $data = [
            'action' => $schema['action'],
            'type' => $schema['authorizable_type'],
            'role_id' => $roleId,
            'uuid' => null,
        ];

        if (isset($schema['authorizable_uuid'])) {
            $data['uuid'] = $schema['authorizable_uuid'];
        }

        $this->tempStore = [];
        $this->deletePermissionChildren($data);

        DB::table('inside_permissions')->where($data)->delete();
        PermissionDeletedEvent::dispatch($data);
    }

    /**
     * Create permission for given role
     */
    public function createPermission(int $roleId, array $schema, bool $addTranslations = true): void
    {
        if (env('INSIDE_DEBUGING_PERMISSIONS', false) == true) {
            Log::debug(__('createPermission :roleId', ['roleId' => $roleId]), $schema);
        }
        $data = [
            'action' => $schema['custom'] ?? $schema['action'],
            'type' => $schema['authorizable_type'],
            'role_id' => $roleId,
            'uuid' => $schema['authorizable_uuid'] ?? null,
        ];

        if ($data['action'] == 'create' && ! $data['uuid']) {
            try {
                DB::table('inside_permissions')->insert($data);
            } catch (QueryException) {
                // Do nothing if duplicated content
            }
            $this->tempStore = [];
            $this->createPermissionChildren($data);

            PermissionCreatedEvent::dispatch($data);

            return;
        }

        $this->tempStore = [];

        // Insert children permissions
        $this->createPermissionChildren($data);

        try {
            // Insert current permission
            DB::table('inside_permissions')->insert($data);

            // TODO: this is ugly
            if ($schema['authorizable_uuid'] && $schema['authorizable_type'] === type_to_class('main_menus') && $addTranslations) {
                $menuLink = call_user_func(type_to_class('main_menus').'::find', $schema['authorizable_uuid']);
                foreach (list_languages() as $language) {
                    if ($language === $menuLink->langcode) {
                        continue;
                    }

                    $translation = $menuLink->getTranslationIfExist($language);

                    if ($translation->langcode === $menuLink->langcode) {
                        continue;
                    }

                    $schema['authorizable_uuid'] = $translation->uuid;
                    $this->createPermission($roleId, $schema, false);
                }
            }

            PermissionCreatedEvent::dispatch($data);
        } catch (QueryException) {
            // Do nothing if duplicated content
        }
    }

    protected function deletePermissionChildren(array $data): void
    {
        if (env('INSIDE_DEBUGING_PERMISSIONS', false) == true) {
            Log::debug(__('deletePermissionChildren'), $data);
        }
        $key = 'uuid';
        $parentKey = 'pid';
        $parentUuid = $data['uuid'] ?? null;
        $parentType = $data['type'];

        if (Str::startsWith($data['type'], 'Inside\Menu\\')) {
            $table = menu_class_to_table($parentType);
            $parentKey = 'parent_uuid';
        } else {
            $table = class_to_table($parentType);
        }

        $query = DB::table($table)->select([$key]);

        if ($parentUuid) {
            $query->where($parentKey, $parentUuid)->where($key, '<>', $parentUuid);
        } else {
            $query->where(function ($query) use ($parentKey) {
                $query->whereNull($parentKey)->orWhere($parentKey, '');
            });
        }

        $uuids = $query->get()->pluck($key)->toArray();

        foreach ($uuids as $uuid) {
            $data['uuid'] = $uuid;

            try {
                DB::table('inside_permissions')->where($data)->delete();
                PermissionDeletedEvent::dispatch($data);
            } catch (QueryException) {
                // Do nothing if duplicated content
            }

            if (! in_array($data['uuid'], $this->tempStore)) {
                $this->tempStore[] = $data['uuid'];
                $this->deletePermissionChildren($data);
            }
        }

        // check inheritance
        if ($data['action'] == 'read' && in_array($parentType, $this->categorizableClasses)) {
            // Delete all contents with this field that has empty pivot
            foreach ($this->schemaService->getFieldNamesThatReferenceType($parentType) as $type => $fieldNames) {
                foreach ($fieldNames as $fieldName) {
                    $options = $this->schemaService->getFieldOptions($type, $fieldName);
                    $required = $options['required'] ?? false;
                    if ($required === false) {
                        // Remove all content with empty categorizable content type ...
                        // TODO: count pivot ... & delete if empty
                    }
                }
            }
            $this->deleteInheritedPermission($data);
        }
    }

    protected function deleteInheritedPermission(array $data): void
    {
        if (env('INSIDE_DEBUGING_PERMISSIONS', false) == true) {
            Log::debug(__('deleteInheritedPermission'), ['data' => $data, 'tempStore' => count($this->tempStore)]);
        }
        $parentUuid = $data['uuid'] ?? null;
        $parentType = $data['type'];

        // Be sure that there is not a schema that grant this content later !
        if (
            $data['uuid'] !== null && DB::table('inside_permissions_schema')
                ->join(
                    'inside_roles_permissions_schema',
                    'inside_permissions_schema.id',
                    'inside_roles_permissions_schema.permission_schema_id'
                )
                ->where(
                    'inside_permissions_schema.authorizable_type',
                    $data['type']
                )
                ->where(
                    'inside_permissions_schema.authorizable_uuid',
                    $data['uuid']
                )
                ->where('inside_permissions_schema.action', $data['action'])
                ->where(
                    'inside_roles_permissions_schema.role_id',
                    $data['role_id']
                )->where('inside_permissions_schema.invert', false)->exists()
        ) {
            return; // This permission will be grant later don't delete it !
        }

        $pivots = DB::table('inside_pivots')
            ->where('related_uuid', $parentUuid)
            ->when($parentUuid, function ($query) use ($parentUuid) {
                $query->where('parent_uuid', '<>', $parentUuid);
            })
            ->when($parentType, function ($query) use ($parentType) {
                $query->where(
                    'parent_type',
                    '<>',
                    $parentType
                ); // already done by deleteChildrenPermission
            })
            ->whereIn(
                'parent_type',
                $this->permissibleClasses
            ) // non permissible content are always allowed !
            ->where('parent_type', '<>', Users::class)
            ->get(); // We don't don't delete user fields permission

        foreach ($pivots as $pivot) {
            $data = [
                'role_id' => $data['role_id'],
                'uuid' => $pivot->parent_uuid,
                'type' => $pivot->parent_type,
                'action' => $data['action'],
            ];
            // Check this is the only one
            if (
                DB::table('inside_pivots')
                    ->where('parent_uuid', $pivot->parent_uuid)
                    ->where('related_field', $pivot->related_field)
                    ->count()
                != 1
            ) {
                // Check at least one category is allowed via schema
                $relatedUuids = DB::table('inside_pivots')
                    ->where('parent_uuid', $pivot->parent_uuid)
                    ->where('related_field', $pivot->related_field)
                    ->pluck('related_uuid');
                foreach ($relatedUuids as $relatedUuid) {
                    if ($this->allowedBySchema($data['role_id'], $data['action'], class_to_type($parentType), $relatedUuid)) {
                        continue 2; // IMPORTANT NOTE: Permission is removed only if the forbidden related relation is the only
                        // one.
                        // Example: a contact that is in a forbidden content category is forbidden as well only if the forbidden news
                        // category is the only one
                    }
                }
            }

            DB::table('inside_permissions')->where($data)->delete();
            PermissionDeletedEvent::dispatch($data);

            if (! in_array($data['uuid'], $this->tempStore)) {
                $this->tempStore[] = $data['uuid'];
                $this->deleteInheritedPermission($data);
            }
            // We had a restriction, Make sure null is not present
            $data['uuid'] = null;
            DB::table('inside_permissions')->where($data)->delete();
        }
    }

    /**
     * Create permission children
     */
    protected function createPermissionChildren(array $data): void
    {
        if (env('INSIDE_DEBUGING_PERMISSIONS', false) == true) {
            Log::debug(__('createPermissionChildren'), $data);
        }
        $key = 'uuid';
        $parentKey = 'pid';
        $parentUuid = $data['uuid'] ?? null;
        $parentType = $data['type'];

        if (Str::startsWith($data['type'], 'Inside\Menu\\')) {
            $table = menu_class_to_table($parentType);
            $parentKey = 'parent_uuid';
        } else {
            $table = class_to_table($parentType);
        }

        $query = DB::table($table)->select([$key]);

        if ($parentUuid !== null) {
            $query->where($parentKey, $parentUuid)->where($key, '<>', $parentUuid);
        } else {
            $query->where(function ($query) use ($parentKey) {
                $query->whereNull($parentKey)->orWhere($parentKey, '');
            });
        }

        $uuids = $query->get()->pluck($key)->toArray();

        foreach ($uuids as $uuid) {
            $data['uuid'] = $uuid;

            try {
                DB::table('inside_permissions')->insert($data);
                PermissionCreatedEvent::dispatch($data);
            } catch (QueryException) {
                // Do nothing if duplicated content
            }

            if (! in_array($data['uuid'], $this->tempStore)) {
                $this->tempStore[] = $data['uuid'];
                $this->createPermissionChildren($data);
            }
        }
    }

    /**
     * Get the root role to be sure of the permission inheritance
     */
    public function getRootRole(Role $role): Role
    {
        if ($role->pid) {
            return $this->getRootRole($role->parent);
        }

        return $role;
    }

    /**
     * List user add permissions
     */
    public function getAllowedCreationTypesForUser(User $user): array
    {
        $add = [];

        if ($this->isSuperAdmin($user)) {
            return InsideSchema::getContentTypes();
        }

        if (! $user->roles) {
            return [];
        }

        foreach ($this->permissibleClasses as $type) {
            $type = class_to_type($type);
            if ($this->allowed('create', $type, null, $user)) {
                $add[] = $type;
            }
        }

        $types = InsideSchema::getContentTypes();
        $customInAddList = config('permission.in_add_list', []);
        if (! empty($customInAddList)) {
            foreach ($types as $type) {
                $can = true;

                foreach ($customInAddList as $canAdd) {
                    if ($canAdd && is_callable($canAdd)) {
                        $can &= $canAdd(type_to_class($type));
                    }
                }

                if ($can && ! array_search($type, $add)) {
                    $add[] = $type;
                }
            }
        }

        return $add;
    }

    /**
     * List user custom permissions
     */
    public function customList(User $user): array
    {
        if (! $user->roles) {
            return [];
        }

        return DB::table('inside_permissions')->select('action')
            ->where('type', 'custom')->whereNull('uuid')->where(function (
                $query
            ) use ($user) {
                $query->whereIn(
                    'role_id',
                    $user->roles->pluck('id')->toArray()
                );
                $query->orWhere('user_uuid', $user->uuid);
            })->get()->pluck('action')->toArray();
    }

    public function getBackofficeEntriesForRole(Role|int $role): Collection
    {
        if (is_int($role)) {
            $role = Role::find($role);
        }

        if (static::isSystemV2Enabled()) {
            /** @var Role $role */

            return RolePrivilegesService::of($role)->getBackofficePrivileges()->map(fn (BackofficePrivilegeDto $dto) => [
                'name' => strtolower($dto->getName()),
                'label' => BackofficeEnum::label($dto->getName(), auth()->user()->langcode),
                'permission' => $dto->isAuthorized(),
            ]);
        }

        $this->loadBackofficeEntriesAccessCache();

        return Inside::getAllBackofficeEntries()->transform(fn ($entry) => [
            'name' => $entry,
            'label' => __('permission.backoffice_entries.'.$entry),
            'permission' => $this->backofficeEntriesAccessCache[$role->id]?->contains($entry),
        ]);
    }

    /**
     * Check user has access to backoffice entry $entry
     * @deprecated kept for backward compatibility
     */
    public function userCanAccessBackofficeEntry(string $entry, ?User $user = null): bool
    {
        return (static::isSystemV2Enabled())
            ? $this->getPermissionUserFromAnyUser($user)->hasBackofficeAccessTo($entry)
            : $this->backofficeAccessibleEntries($this->getPermissionUserFromAnyUser($user))->contains($entry);
    }

    public function updateBackofficeEntriesForRole(Role | int $role, array $entries): void
    {
        if (is_int($role)) {
            $role = Role::find($role);
        }
        DB::table('inside_permissions')->where(
            'type',
            'backoffice'
        )->where('role_id', $role->id)->delete();
        foreach ($entries as $entry) {
            Permission::create([
                'type' => 'backoffice',
                'action' => Str::snake('access_'.$entry),
                'role_id' => $role->id,
            ]);
        }
        $this->forgetCachedBackofficeEntriesAccess();
    }

    /**
     * List dashboard custom permissions
     * @deprecated kept for backward compatibility
     */
    public function backofficeAccessibleEntries(string|User|Users|null $user = null): Collection
    {
        if (static::isSystemV2Enabled()) {
            return $this->getPermissionUserFromAnyUser($user)->getAuthorizedBackofficePrivileges();
        }

        $user = $this->getPermissionUserFromAnyUser($user);

        if ($user->isSuperAdmin()) {
            // Super admin get permission entry
            return Inside::getAllBackofficeEntries()->merge([
                'permission',
            ]);
        }

        $this->loadBackofficeEntriesAccessCache();

        return $this->backofficeEntriesAccessCache->only($user->roles->pluck('id'))->flatten();
    }

    public function forgetCachedBackofficeEntriesAccess(): bool
    {
        $this->backofficeEntriesAccessCache = null;

        return Cache::forget('inside.permission.backoffice.entries.access');
    }

    /**
     * @throws ModelNotFoundException
     */
    public function getPermissionUserFromAnyUser(string|User|Users|null $user = null): User
    {
        if (is_null($user)) {
            /** @var User $user */
            $user = Auth::user();
        }
        if (is_string($user)) {
            return User::find($user);
        } elseif ($user instanceof User) {
            return $user;
        } elseif ($user instanceof Users) {
            return User::find($user->uuid);
        } else {
            throw new ModelNotFoundException();
        }
    }

    protected function loadBackofficeEntriesAccessCache(): void
    {
        if (! is_null($this->backofficeEntriesAccessCache)) {
            return;
        }

        $this->backofficeEntriesAccessCache = Cache::remember(
            'inside.permission.backoffice.entries.access',
            1440,
            function () {
                $backofficeEntriesAccess = collect();
                Role::each(function ($role) use ($backofficeEntriesAccess) {
                    $backofficeEntriesAccess[$role->id] = Inside::getAllBackofficeEntries()->filter(
                        function ($entry) use ($role) {
                            return $role->name == 'super_administrator' ||
                                DB::table('inside_permissions')->where('type', 'backoffice')
                                    ->where('action', Str::snake('access_'.$entry))
                                    ->where('role_id', $role->id)->exists();
                        }
                    );
                });

                return $backofficeEntriesAccess;
            }
        );
    }

    public function removeChildrenPermission(Content $content): void
    {
        $roleIds = DB::table('inside_roles')->get()->pluck('id');

        // At this point permission set by schema is done, we can use it!
        foreach ($roleIds as $roleId) {
            if (
                RoleService::can(
                    'read',
                    $content,
                    $roleId
                )
            ) {
                continue; // This is allowed, no need to remove permission on children
            }
            // Get children
            foreach ($content->getChildrenIfExist()->pluck('uuid') as $uuid) {
                // This content has a forbidden parent, it should be forbidden as well !
                \Inside\Permission\Facades\Permission::deletePermission(
                    $roleId,
                    [
                        'authorizable_uuid' => $uuid,
                        'authorizable_type' => get_class($content),
                        'action' => 'read',
                    ]
                );
            }
        }
    }

    public function removePermissionByInheritance(Content $content): void
    {
        $roleIds = DB::table('inside_roles')->get()->pluck('id');

        // At this point permission set by schema is done, we can use it!
        foreach ($roleIds as $k => $roleId) {
            if (! RoleService::can('read', $content, $roleId)) {
                unset($roleIds[$k]);
            }
        }

        // Check & get categorizable field
        $fieldNames = InsideSchema::getFieldListing(
            class_to_type($content),
            function ($fieldOptions) {
                if ($fieldOptions['type'] != 'reference') {
                    return false;
                }
                $target = Arr::first($fieldOptions['options']['target']);
                if ($target === null) {
                    return false;
                }

                return in_array(
                    type_to_class($target),
                    $this->categorizableClasses
                );
            }
        );

        foreach ($fieldNames as $fieldName) {
            $targetContent = $content->{Str::camel($fieldName)};
            if ($targetContent === null || $targetContent->isEmpty()) {
                continue;
            }
            $fieldOptions = InsideSchema::getFieldOptions(class_to_type($content), $fieldName);
            if ($fieldOptions['cardinality'] == -1 && $targetContent->count() > 1) {
                continue;
            }
            $targetContent = Arr::first($targetContent);
            foreach ($roleIds as $roleId) {
                if (! RoleService::can('read', $targetContent, $roleId)) {
                    $this->deletePermission($roleId, [
                        'authorizable_uuid' => $content->uuid,
                        'authorizable_type' => get_class($content),
                        'action' => 'read',
                        'children' => [],
                    ]);
                }
            }
        }
    }

    public function allowedBySchema(int $roleId, string $action, string $type, string $uuid, bool $forChildren = false): bool
    {
        $query = DB::table('inside_permissions_schema')
            ->join(
                'inside_roles_permissions_schema',
                'inside_permissions_schema.id',
                '=',
                'inside_roles_permissions_schema.permission_schema_id'
            )
            ->where('inside_roles_permissions_schema.role_id', $roleId)
            ->where('inside_roles_permissions_schema.is_content_specific', 0)
            ->where('inside_permissions_schema.action', $action)
            ->where('inside_permissions_schema.authorizable_type', type_to_class($type))
            ->where('inside_permissions_schema.invert', 0)
            ->where(
                fn (Builder $query) => $query->whereNull('inside_permissions_schema.authorizable_uuid')
                    ->orWhere('inside_permissions_schema.authorizable_uuid', $uuid)
            );

        if ($query->exists()) {
            return true;
        }

        // No schema found, let's try with its parent
        $query = call_user_func(type_to_class($type).'::withoutGlobalScopes');
        $content = $query->find($uuid);
        if (is_null($content->parent)) {
            return false;
        }
        if (Cache::has("$roleId-$action-$type-{$content->parent->uuid}")) {
            return false; // Avoid loop
        }
        Cache::forever("$roleId-$action-$type-{$content->parent->uuid}", now());

        return $this->allowedBySchema($roleId, $action, $type, $content->parent->uuid);
    }
}
