<?php

namespace Inside\Permission\Services;

use Exception;
use Illuminate\Database\QueryException;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Inside\Content\Models\Content;
use Inside\Content\Models\Field;
use Inside\Content\Models\Model;
use Inside\Menu\Services\MenuService;
use Inside\Permission\Exodus\Models\Privileges\ContentSpecificPrivilege;
use Inside\Permission\Exodus\Models\Role;
use Inside\Permission\Models\PermissionSchema;

/**
 * Service handle any permission schema
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class PermissionSchemaService
{
    public const DEFAULT_LANGUAGE = 'en';

    /**
     * User langcode
     *
     * @var string
     */
    public string $langcode = self::DEFAULT_LANGUAGE;

    /**
     * The super admin role
     *
     * @var Role|null
     */
    public $superAdminRole = null;

    private PermissionHelperService $permissionHelper;

    /**
     * Construct the service
     */
    public function __construct()
    {
        $user = Auth::user();

        if ($user) {
            $this->langcode = $user->langcode ?? self::DEFAULT_LANGUAGE;
        }

        $this->superAdminRole = Role::where('name', 'super_administrator')
            ->first();

        $this->permissionHelper = PermissionHelperService::getInstance();
    }

    /**
     * Generate permission tree
     *
     * @param  array  $roleIds
     *
     * @return array
     */
    public function tree(array $roleIds): array
    {
        $permissions = [];
        $models = Model::all();
        $types = [
            'weight' => 20,
            'type' => 'types',
            'permissions' => [],
        ];
        $categories = [
            'weight' => 25,
            'type' => 'categories',
            'permissions' => [],
        ];

        foreach ($models as $model) {
            $hasWeight = Field::whereHas(
                'model',
                function ($query) use ($model) {
                    $query->whereClass($model);
                }
            )->whereName('weight')->exists();

            if (isset($model->options['permissible'])
                && $model->options['permissible']
                && ! in_array(
                    class_to_type($model->class),
                    config('permission.hidden_types', [])
                )
            ) {
                if (Str::endsWith($model->class, 'Menus')) {
                    continue;
                }

                $actions = $this->getActionsForRoles($roleIds, $model);
                $permission = [
                    'title' => $model->options['title'][$this->langcode] ?? '',
                    'type' => $model->options['name'],
                    'permission' => $actions,
                    'children' => [],
                ];
                if (! $model->options['categorizable']) {
                    $types['permissions'][] = $permission;
                } else {
                    $permission['children'] = $this->contentTree(
                        $model,
                        $roleIds,
                        null,
                        $hasWeight
                    );
                    $categories['permissions'][] = $permission;
                }
            }
        }

        $menus = [
            'header',
            // TODO: add footer in this array when the new footer menu is ready to be launched
            //            'footer'
        ];

        foreach ($menus as $menuName) {
            $permission = [
                'weight' => 5,
                'type' => $menuName,
                'title' => __('permission.title.'.$menuName),
                'permission' => [],
                'children' => $this->menuTree($menuName, $roleIds),
            ];

            $permissions[] = $permission;
        }

        $permissions[] = $types;
        $permissions[] = $categories;

        return $permissions;
    }

    /**
     * Check role can perform action
     *
     * @param  array  $roleIds
     * @param  Model  $model
     * @param  mixed  $content
     *
     * @return array|null
     */
    public function getActionsForRoles(
        array $roleIds,
        Model $model,
        $content = null
    ): ?array {
        $actions = [
            'read' => null,
            'create' => null,
            'update' => null,
            'delete' => null,
        ];

        if ($superAdminActions = $this->getSuperAdminActions(
            $roleIds,
            $model,
            $content
        )
        ) {
            return $superAdminActions;
        }

        $schemas = DB::table('inside_permissions_schema')->select([
            'action',
            'authorizable_type',
            'authorizable_uuid',
            'invert',
            'children',
        ])->where('authorizable_type', $model->class)
            ->where('authorizable_uuid', $content ? $content->uuid : null)
            ->whereIn('action', ['create', 'read', 'update', 'delete'])
            ->join('inside_roles_permissions_schema', function ($join) {
                $join->on(
                    'inside_roles_permissions_schema.permission_schema_id',
                    '=',
                    'inside_permissions_schema.id'
                );
            })->whereIn('inside_roles_permissions_schema.role_id', $roleIds)
            ->orderByDesc('invert')->get();

        foreach ($schemas as $schema) {
            $actions[$schema->action] = (array) $schema;
        }

        return $actions;
    }

    /**
     * Get actions if user is super administrator
     *
     * @param  array  $roleIds
     * @param  mixed  $model
     * @param  mixed  $content
     *
     * @return array|null
     */
    public function getSuperAdminActions(
        array $roleIds,
        $model,
        $content
    ): ?array {
        if ($this->superAdminRole
            && in_array($this->superAdminRole->id, $roleIds)
        ) {
            return [
                'read' => [
                    'action' => 'read',
                    'children' => true,
                    'invert' => false,
                    'authorizable_type' => $model->class,
                    'authorizable_uuid' => isset($content) ? $content->uuid
                        : null,
                ],
                'create' => [
                    'action' => 'create',
                    'children' => true,
                    'invert' => false,
                    'authorizable_type' => $model->class,
                    'authorizable_uuid' => isset($content) ? $content->uuid
                        : null,
                ],
                'update' => [
                    'action' => 'update',
                    'children' => true,
                    'invert' => false,
                    'authorizable_type' => $model->class,
                    'authorizable_uuid' => isset($content) ? $content->uuid
                        : null,
                ],
                'delete' => [
                    'action' => 'delete',
                    'children' => true,
                    'invert' => false,
                    'authorizable_type' => $model->class,
                    'authorizable_uuid' => isset($content) ? $content->uuid
                        : null,
                ],
            ];
        }

        return null;
    }

    /**
     * Check role can perform action
     *
     * @param array $roleIds
     * @param string $menuName
     * @param array|null $menuLink
     * @return array|null
     */
    public function getMenuActionsForRoles(
        array $roleIds,
        string $menuName,
        ?array $menuLink = null
    ): ?array {
        $actions = [
            'read' => null,
        ];

        if ($superAdminActions = $this->getSuperAdminMenuActions(
            $roleIds,
            $menuName,
            $menuLink
        )
        ) {
            return $superAdminActions;
        }

        $schemas = DB::table('inside_permissions_schema')->select([
            'action',
            'authorizable_type',
            'authorizable_uuid',
            'invert',
            'children',
        ])->where('authorizable_type', menu_type_to_class($menuName))
            ->where('authorizable_uuid', $menuLink ? (string) $menuLink['uuid'] : null)
            ->whereIn('action', ['read'])
            ->join('inside_roles_permissions_schema', function ($join) {
                $join->on(
                    'inside_roles_permissions_schema.permission_schema_id',
                    '=',
                    'inside_permissions_schema.id'
                );
            })->whereIn('inside_roles_permissions_schema.role_id', $roleIds)
            ->orderByDesc('invert')->get();

        foreach ($schemas as $schema) {
            $actions[$schema->action] = (array) $schema;
        }

        return $actions;
    }

    /**
     * Get actions if user is super administrator
     *
     * @param array $roleIds
     * @param string $menuName
     * @param array|null $menuLink
     * @return array|null
     */
    public function getSuperAdminMenuActions(
        array $roleIds,
        string $menuName,
        ?array $menuLink
    ): ?array {
        if ($this->superAdminRole
            && in_array($this->superAdminRole->id, $roleIds)
        ) {
            return [
                'read' => [
                    'action' => 'read',
                    'children' => true,
                    'invert' => false,
                    'authorizable_type' => menu_type_to_class($menuName),
                    'authorizable_uuid' => isset($menuLink) ? $menuLink['uuid']
                        : null,
                ],
                'create' => [
                    'action' => 'create',
                    'children' => true,
                    'invert' => false,
                    'authorizable_type' => menu_type_to_class($menuName),
                    'authorizable_uuid' => isset($menuLink) ? $menuLink['uuid']
                        : null,
                ],
                'update' => [
                    'action' => 'update',
                    'children' => true,
                    'invert' => false,
                    'authorizable_type' => menu_type_to_class($menuName),
                    'authorizable_uuid' => isset($menuLink) ? $menuLink['uuid']
                        : null,
                ],
                'delete' => [
                    'action' => 'delete',
                    'children' => true,
                    'invert' => false,
                    'authorizable_type' => menu_type_to_class($menuName),
                    'authorizable_uuid' => isset($menuLink) ? $menuLink['uuid']
                        : null,
                ],
            ];
        }

        return null;
    }

    /**
     * List menu item
     *
     * @param  Model  $model
     * @param  array  $roleIds
     * @param  string|null  $parent
     * @param  bool  $hasWeight
     *
     * @return array
     */
    public function contentTree(
        Model $model,
        array $roleIds,
        ?string $parent = null,
        bool $hasWeight = false
    ): array {
        $tree = [];
        $table = class_to_table($model->class);
        $query = DB::table($table)->where('langcode', $this->langcode);

        // TODO : Have mixed null and empty value in DB.
        if (! $parent) {
            $query->where(function ($query) {
                $query->where('pid', '')->orWhereNull('pid');
            });
        } else {
            $query->where('pid', $parent);
        }

        $query->orderBy('title');

        if ($hasWeight) {
            $query->orderBy('weight', 'title');
        }

        $contents = $query->get();

        foreach ($contents as $content) {
            $actions = $this->getActionsForRoles($roleIds, $model, $content);
            $node = [
                'title' => $content->title,
                'uuid' => $content->uuid,
                'type' => $model->options['name'], // @phpstan-ignore-line
                'permission' => $actions,
                'children' => $this->contentTree(
                    $model,
                    $roleIds,
                    $content->uuid,
                    $hasWeight
                ),
            ];

            $tree[] = $node;
        }

        return $tree;
    }

    /**
     * List menu item
     */
    public function menuTree(
        string $menuName,
        array $roleIds,
        ?string $parent = null,
        ?string $locale = null
    ): array {
        $tree = [];

        if (is_null($locale)) {
            $locale = $this->langcode;
        }

        $menuService = new MenuService();
        $menuLinks = $menuService->getMenuLinks($menuName, $locale, true, $parent, false);

        foreach ($menuLinks as $menuLink) {
            $actions = $this->getMenuActionsForRoles($roleIds, $menuName, $menuLink);
            $node = [
                'title' => $menuLink['title'],
                'uuid' => (string) $menuLink['uuid'],
                'type' => $menuName,
                'permission' => $actions,
                'children' => $this->menuTree($menuName, $roleIds, $menuLink['uuid'], $locale),
            ];
            $tree[] = $node;
        }

        return $tree;
    }

    /**
     * Check role can perform action
     */
    public function getActionsForRole(
        int $roleId,
        Model $model,
        ?Content $content = null
    ): array {
        $actions = [
            'read' => null,
            'create' => null,
            'update' => null,
            'delete' => null,
        ];

        $schemas = DB::table('inside_permissions_schema')->select([
            'action',
            'authorizable_type',
            'authorizable_uuid',
            'invert',
            'children',
        ])->where('authorizable_type', $model->class)
            ->where('authorizable_uuid', $content ? $content->uuid : null)
            ->whereIn('action', ['create', 'read', 'update', 'delete'])
            ->join('inside_roles_permissions_schema', function ($join) {
                $join->on(
                    'inside_roles_permissions_schema.permission_schema_id',
                    '=',
                    'inside_permissions_schema.id'
                );
            })->where('inside_roles_permissions_schema.role_id', $roleId)
            ->get();

        foreach ($schemas as $schema) {
            $actions[$schema->action] = (array) $schema;
        }

        return $actions;
    }

    /**
     * Delete permission schema and clean useless ones
     *
     * @param  int  $roleId
     *
     * @return void
     */
    public function deletePermissionsSchemaForRole(int $roleId): void
    {
        DB::table('inside_roles_permissions_schema')->where([
            'role_id' => $roleId,
            'is_content_specific' => false,
        ])->delete();
    }

    /**
     * Create new permission schema and clean useless ones
     *
     * @param  array  $schemas
     * @param  int  $roleId
     *
     * @return void
     */
    public function create(array $schemas, int $roleId): void
    {
        $sync = [];

        foreach ($schemas as $schema) {
            if ($schema['authorizable_type']
                && in_array(
                    $schema['authorizable_type'],
                    ['header', 'footer', 'profile', 'icon']
                )) {
                $schema['authorizable_type'] = menu_type_to_class($schema['authorizable_type']);
            }

            if ($schema['authorizable_type']
                && ! class_exists($schema['authorizable_type'])
            ) {
                $schema['authorizable_type']
                    = type_to_class($schema['authorizable_type']);
            }

            if (! isset($schema['authorizable_uuid'])) {
                $schema['authorizable_uuid'] = null;
            }

            $sync[] = $this->permissionHelper->getOrCreatePermissionSchema($schema);

            if (Str::endsWith($schema['authorizable_type'], 'Menus')
                && $schema['invert'] == 1
            ) {
                // Because performance of array_merge is incredibly slow and memoryvore...
                foreach ($this->handleMenuItems($schema) as $add) {
                    if (! in_array($add, $sync)) {
                        $sync[] = $add;
                    }
                }
            }

            // Because performance of array_merge is incredibly slow and memoryvore...
            foreach ($this->translateSchema($schema) as $add) {
                if (! in_array($add, $sync)) {
                    $sync[] = $add;
                }
            }
        }

        $toInsert = [];

        $existingRelations = DB::table('inside_roles_permissions_schema')
            ->where('role_id', $roleId)->whereIn('permission_schema_id', $sync)
            ->get()->pluck('permission_schema_id');

        foreach ($sync as $id) {
            if ($existingRelations->contains($id)) {
                continue;
            }
            $toInsert[] = [
                'role_id' => $roleId,
                'permission_schema_id' => $id,
            ];
        }

        try {
            DB::table('inside_roles_permissions_schema')->insert($toInsert);
        } catch (QueryException $e) {
            // Do nothing, already attached to this role
        }
    }

    /**
     * Handle menu items
     */
    public function handleMenuItems(array $schema): array
    {
        $table = class_to_table($schema['authorizable_type']);
        $link = DB::table($table)->where('uuid', $schema['authorizable_uuid'])
            ->get()->pluck('link')->first();
        $slugs = DB::table('inside_slugs')->where('slug', $link)->get();
        $sync = [];

        foreach ($slugs as $slug) {
            $class = table_to_class($slug->type);
            $data = [
                'action' => 'read',
                'invert' => true,
                'children' => true,
                'authorizable_type' => $class,
                'authorizable_uuid' => $slug->uuid,
            ];

            $sync[] = $this->permissionHelper->getOrCreatePermissionSchema($data);

            foreach ($this->translateSchema($data) as $add) {
                // Because performance of array_merge is incredibly slow and memoryvore...
                if (! in_array($add, $sync)) {
                    $sync[] = $add;
                }
            }
        }

        return $sync;
    }

    /**
     * Attach translations
     */
    public function translateSchema(array $schema): array
    {
        $sync = [];

        if (! isset($schema['authorizable_uuid'])) {
            return $sync;
        }

        if (Str::startsWith($schema['authorizable_type'], 'Inside\Menu\\')) {
            return $sync;
        }

        $table = class_to_table($schema['authorizable_type']);
        $translations = DB::table($table)->select('uuid')
            ->where('uuid_host', function ($query) use ($table, $schema) {
                $query->select('uuid_host')->from($table)
                    ->where('uuid', $schema['authorizable_uuid']);
            })->where('uuid', '<>', $schema['authorizable_uuid'])->get();

        foreach ($translations as $translation) {
            $translatedSchema = $schema;
            $translatedSchema['authorizable_uuid'] = $translation->uuid;

            $sync[] = $this->permissionHelper->getOrCreatePermissionSchema($translatedSchema);
        }

        return $sync;
    }

    /**
     * Remove useless permissions schema
     *
     * @return void
     */
    public function clean(): void
    {
        try {
            PermissionSchema::doesntHave('roles')->doesntHave('users')
                ->delete();
        } catch (Exception $e) {
        }
    }

    /**
     * @param  Content|string  $contentUuid
     * @param  string|null  $type
     * @param  string  $action
     */
    public function resetContentSpecificPermissions($contentUuid, ?string $type = null, string $action = 'read'): void
    {
        $class = null;
        if ($type !== null) {
            $class = type_to_class($type);
        }
        if ($contentUuid instanceof Content) {
            $class = get_class($contentUuid);
            $contentUuid = $contentUuid->uuid;
        }
        DB::table('inside_permissions_schema')
            ->join(
                'inside_roles_permissions_schema',
                'inside_permissions_schema.id',
                'inside_roles_permissions_schema.permission_schema_id'
            )
            ->where('inside_roles_permissions_schema.is_content_specific', true)
            ->when($class !== null, function ($query) use ($class) {
                $query->where(
                    'inside_permissions_schema.authorizable_type',
                    $class
                );
            })->where('inside_permissions_schema.action', $action)
            ->where('inside_permissions_schema.authorizable_uuid', $contentUuid)
            ->delete();
    }

    /**
     * @param  Content  $content
     */
    public function resetContentSpecificPermissionsV2(Content $content): void
    {
        DB::table(ContentSpecificPrivilege::PIVOT_ROLE_TABLE)
            ->leftJoin(ContentSpecificPrivilege::TABLE, 'content_privilege_id', '=', 'id')
            ->where('uuid', $content->uuid)
            ->select('pivot_id')
            ->delete();
    }

    /**
     * @param  string|Content  $contentUuid
     * @param  string|null  $type
     * @param  string  $action
     *
     * @return bool
     */
    public function hasContentSpecificPermission(
        $contentUuid,
        ?string $type = null,
        string $action = 'read'
    ): bool {
        $class = null;
        if ($type !== null) {
            $class = type_to_class($type);
        }
        if ($contentUuid instanceof Content) {
            $class = get_class($contentUuid);
            $contentUuid = $contentUuid->uuid;
        }

        $query = DB::table('inside_permissions_schema')
            ->join(
                'inside_roles_permissions_schema',
                'inside_roles_permissions_schema.permission_schema_id',
                'inside_permissions_schema.id'
            )->when(
                $class !== null,
                function ($query) use ($class) {
                    $query->where(
                        'inside_permissions_schema.authorizable_type',
                        $class
                    );
                }
            )
            ->where('inside_permissions_schema.authorizable_uuid', $contentUuid)
            ->where('inside_permissions_schema.action', $action);

        $hasContentSpecific = $query->where(
            'inside_roles_permissions_schema.is_content_specific',
            true
        )->exists();

        if ($hasContentSpecific) {
            // check if content specific does not match global permission
            $contentSpecificRoleIds = $query->where(
                'inside_roles_permissions_schema.is_content_specific',
                true
            )->where('invert', false)->orderBy('role_id')
                ->pluck('role_id', 'role_id');

            $nonContentSpecificRoleIds = $query->where('invert', false)
                ->where(
                    'inside_roles_permissions_schema.is_content_specific',
                    false
                )->orderBy('role_id')->pluck('role_id', 'role_id');

            // Check allowed role
            foreach ($contentSpecificRoleIds as $roleId) {
                if ($nonContentSpecificRoleIds->has($roleId)) {
                    unset($contentSpecificRoleIds[$roleId]);
                }
            }
            if ($contentSpecificRoleIds->isEmpty()) {
                return false;
            }
        }

        return $hasContentSpecific;
    }

    /**
     * @param  Content  $content
     *
     * @return bool
     */
    public function hasContentSpecificPermissionV2(Content $content): bool
    {
        if (! config('permission.specific_permissions_enabled') || blank($content->uuid_host)) {
            return false;
        }

        return $content->hasSpecificPrivilege();
    }

    public function getContentSpecificPermissions(Content $content): Collection
    {
        if (! config('permission.specific_permissions_enabled') || blank($content->uuid_host)) {
            return collect();
        }

        return Role::query()
            ->whereHas('contentSpecificPrivileges', fn ($query) => $query->where('uuid', $content->uuid))
            ->pluck('id');
    }

    public function getExistingMenuSchema(string $roleId, string $langcode, array $schemas): array
    {
        $menus = [
            'header',
        ];

        foreach ($menus as $menuName) {
            $tree = $this->menuTree($menuName, [$roleId], null, $langcode);
            if (! array_key_exists($menuName, $schemas)) {
                $schemas[$menuName] = [];
            }
            $this->convertMenuSchemaToPermissionSchema($tree, $schemas[$menuName]);
        }

        return $schemas;
    }

    protected function convertMenuSchemaToPermissionSchema(array $schema, array &$permissions): void
    {
        foreach ($schema as $menuSchema) {
            if (! isset($menuSchema['permission']['read'])) {
                continue;
            }
            $permission = [
                'action' => 'read',
                'authorizable_type' => menu_class_to_type($menuSchema['permission']['read']['authorizable_type']),
                'authorizable_uuid' => $menuSchema['permission']['read']['authorizable_uuid'],
                'children' => (bool) $menuSchema['permission']['read']['children'],
                'invert' => (bool) $menuSchema['permission']['read']['invert'],
            ];
            if (is_array($menuSchema['children'])) {
                $this->convertMenuSchemaToPermissionSchema($menuSchema['children'], $permissions);
            }
            $permissions[] = $permission;
        }
    }
}
