<?php

namespace Inside\Permission\Models;

use Event;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Notifications\DatabaseNotificationCollection;
use Illuminate\Support\Facades\Cache;
use Inside\Notify\Concerns\NotifiableRole;
use Inside\Permission\Events\RoleCreatedEvent;
use Inside\Permission\Events\RoleDeletedEvent;
use Inside\Permission\Events\RoleUpdatedEvent;
use Inside\Permission\Exodus\Models\Role as ExodusRole;
use Inside\Permission\Models\Traits\HasBackofficeEntriesAccess;
use Inside\Permission\Models\Traits\RefreshBackofficeEntriesCache;

/**
 * Inside role model
 *
 * @property-read Collection<User> $users
 * @property int $id
 * @property ?int $pid
 * @property ?string $condition
 * @property-read ?self $parent
 * @property string $name
 * @property bool $is_automatic
 * @property DatabaseNotificationCollection $notifications
 *
 * @method static Builder|self selectRaw(string $expression, array $bindings = [])
 * @method static \Inside\Database\Eloquent\Builder|Role where($column, $operator = null, $value = null, $boolean = 'and')
 * @method static Role firstOrFail($columns = ['*'])
 *
 * @mixin Builder
 * @mixin \Eloquent
 */
class Role extends ExodusRole
{
    use NotifiableRole;
    use HasBackofficeEntriesAccess;
    use RefreshBackofficeEntriesCache;

    /**
     * The table without the timestamps.
     *
     * @var bool
     */
    public $timestamps = false;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'inside_roles';

    /**
     * The attributes that are guarded.
     *
     * @var array
     */
    protected $guarded = ['id'];

    /**
     * Fillable properties
     *
     * @var array
     */
    protected $fillable = [
        'name',
        'pid',
        'role_category_id',
        'is_automatic',
        'condition',
        'weight',
        'type',
    ];

    protected $appends = [
        'label',
    ];

    /**
     * The attributes that should be casted to native types.
     *
     * @var array
     */
    protected $casts = [
        'is_automatic' => 'boolean',
    ];

    protected $hidden = ['pivot'];

    public static function boot()
    {
        parent::boot();

        self::created(
            function ($model) {
                Event::dispatch(new RoleCreatedEvent($model));
            }
        );

        self::updated(
            function ($model) {
                Cache::forget('inside_permission.roles.count.'.$model->id);
                Event::dispatch(new RoleUpdatedEvent($model));
            }
        );

        self::deleted(
            function ($model) {
                Cache::forget('inside_permission.roles.count.'.$model->id);
                Event::dispatch(new RoleDeletedEvent($model));
            }
        );
    }

    /**
     * Get all role permissions.
     *
     * @return BelongsToMany
     */
    public function permissionSchemas(): BelongsToMany
    {
        return $this->belongsToMany(
            PermissionSchema::class,
            'inside_roles_permissions_schema',
            'role_id',
            'permission_schema_id'
        );
    }

    /**
     * Get all roles users.
     *
     * @return BelongsToMany
     */
    public function users(): BelongsToMany
    {
        return $this->belongsToMany(
            \Inside\Authentication\Models\User::class,
            'inside_users_roles',
            'role_id',
            'user_uuid'
        );
    }

    /**
     * Get children for this entity
     *
     * @return HasMany
     */
    public function children(): HasMany
    {
        return $this->hasMany(
            self::class,
            'pid',
            'id'
        );
    }

    /**
     * Get role parent
     *
     * @return HasOne
     */
    public function parent(): HasOne
    {
        return $this->hasOne(
            self::class,
            'id',
            'pid'
        );
    }

    /**
     * @return string
     */
    public function getLabelAttribute(): string
    {
        return \Inside\Permission\Facades\Role::getHumanName($this->name);
    }
}
