<?php

namespace Inside\Permission\Listeners;

use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Support\Facades\DB;
use Inside\Content\Events\ContentCreatedEvent;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Model;
use Inside\Permission\Facades\PermissionSchema;
use Inside\Permission\Services\PermissionHelperService;

/**
 * Class AutoAddCreatePermissionOnSpecialContents
 *
 * On content creation, auto add create & view permission if content is special
 */
class AutoAddCreatePermissionOnSpecialContents implements ShouldQueue
{
    public function handle(ContentCreatedEvent $event): void
    {
        $model = $event->model;
        $type = class_to_type($model);
        $helper = PermissionHelperService::getInstance();
        $permissibles = $helper->getPermissible();

        if (! in_array(get_class($model), $permissibles)) {
            return;
        }
        $modelInfos = Schema::getModelOptions($type);
        if (! isset($modelInfos['categorizable']) || ! $modelInfos['categorizable']) {
            return;
        }

        $modelInfos = Model::where('class', get_class($model))->first();

        $hasSpecificPermission = DB::table('inside_roles_permissions_schema')->join(
            'inside_permissions_schema',
            function ($join) {
                $join->on(
                    'inside_permissions_schema.id',
                    '=',
                    'inside_roles_permissions_schema.permission_schema_id'
                );
            }
        )->where(
            [
                'inside_roles_permissions_schema.is_content_specific' => true,
                'inside_permissions_schema.authorizable_uuid' => $model->uuid,
            ]
        )->exists();

        if ($hasSpecificPermission) {
            return;
        }

        // Auto create permission
        $contentQuery = call_user_func(type_to_class($type).'::withoutGlobalScopes');
        $otherContents = $contentQuery->where(type_to_table($type).'.uuid', '<>', $model->uuid)->get();

        $roleIds = DB::table('inside_roles')->get()->pluck('id');
        foreach ($roleIds as $roleId) {
            // A content is special if it is a permissible, categorizable content and all contents have create permission
            // But same content do not have create NULL permission
            $isSpecial =
                ! $otherContents->isEmpty() && null !== PermissionSchema::getActionsForRoles([$roleId], $modelInfos);
            if (! $isSpecial) {
                continue;
            }
            foreach ($otherContents as $content) {
                $otherActions = PermissionSchema::getActionsForRoles([$roleId], $modelInfos, $content);
                if (! array_key_exists('create', $otherActions) || $otherActions['create'] === null) {
                    $isSpecial = false;
                    break;
                }
            }

            if (! $isSpecial) {
                continue;
            }
            // Yeaaa ! we got a special content that need to be created with 'create' permission
            $permission = [
                'action' => 'create',
                'children' => true,
                'invert' => false,
                'authorizable_type' => get_class($model),
                'authorizable_uuid' => $model->uuid,
            ];

            // Save permission schema
            PermissionSchema::create([$permission], $roleId);

            $permission = [
                'role_id' => $roleId,
                'action' => 'create',
                'type' => get_class($model),
                'uuid' => $model->uuid,
            ];

            // Save permission
            try {
                DB::table('inside_permissions')->insert($permission);
            } catch (\Exception $e) {
            }
        }
    }
}
