<?php

declare(strict_types=1);

namespace Inside\Permission\Http\Middlewares;

use Closure;
use Illuminate\Auth\Access\AuthorizationException;
use Illuminate\Database\QueryException;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Inside\Content\Facades\ContentHelper;
use Inside\Content\Facades\Schema;
use Inside\Content\Facades\Schema as InsideSchema;
use Inside\Content\Models\Content;
use Inside\Permission\Facades\Permission;
use Inside\Permission\Facades\PermissionSchema;
use Inside\Permission\Facades\Role as RoleService;
use Symfony\Component\HttpFoundation\Response;

/**
 * Restrict form query
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
final class ContentFormMiddleware
{
    public function handle(Request $request, Closure $next): mixed
    {
        if (! config('permission.specific_permissions_enabled', false)) {
            return $next($request); // Disable specific permission
        }

        return (Permission::isSystemV2Enabled())
            ? $this->handleV2($request, $next)
            : $this->handleV1($request, $next);
    }

    protected function handleV1(Request $request, Closure $next): mixed
    {
        $method = $request->getMethod();
        $path = $request->path();
        $roles = null;
        $uuids = [];

        if (
            strpos($path, 'api/v1/content/') == 0
            && in_array($method, ['POST', 'PUT'])
            && $request->has('roles')
        ) {
            $roles = $request->get('roles');
            unset($request['roles']);
        }

        $response = $next($request);
        if ($response instanceof Response && $response->getStatusCode() != 200) {
            return $response;
        }

        $user = Auth::user();

        // Save roles
        if (str_starts_with($path, 'api/v1/content/') && in_array($method, ['POST', 'PUT'])) {
            /** @var array|null $content */
            $content = json_decode_response($response);
            if ($content === null || ! isset($content['uuid'])) {
                return $response;
            }

            $uuid = $content['uuid'];
            $class = type_to_class($request->get('bundle'));

            if (! $this->hasRoleField($class)) {
                return $response;
            }

            // Get all language uuids
            $content = call_user_func($class.'::find', $uuid);
            if (! $content) {
                return $response;
            }
            $uuids = call_user_func($class.'::query')
                ->where('uuid_host', $content->uuid_host)->pluck('uuid');

            if ($roles === null) {
                // No custom permission ( reset in case it was set before )
                PermissionSchema::resetContentSpecificPermissions($uuid, $request->get('bundle'));

                // Means reinit default permission
                foreach ($uuids as $uuid) {
                    // Permission has change, rebuild cache
                    $content = call_user_func($class.'::find', $uuid);
                    if ($content !== null) {
                        $content->touch();
                    }
                }
            }
        }

        if ($roles) {
            preg_match('/^api\/v1\/content\/(.*)?$/', $path, $match);

            if (! isset($match[1])) {
                return $response;
            }

            $exploded = explode('/', $match[1]);
            $class = type_to_class($exploded[0]);

            if (! $this->hasRoleField($class)) {
                return $response;
            }

            foreach ($uuids as $uuid) {
                $data = [
                    'authorizable_type' => $class,
                    'authorizable_uuid' => $uuid,
                    'action' => 'read',
                    'invert' => false,
                    'children' => true,
                ];

                foreach ($roles as $role) {
                    if (! $role['type'] == 'role') {
                        continue;
                    }
                    $permissionSchemaGranter = $permissionSchemaUngranter = $data;
                    $invert = ! ((bool) $role['value']);

                    $permissionSchemaGranter['invert'] = false;
                    $permissionSchemaUngranter['invert'] = true;

                    // Load granter
                    $permissionSchemaGranterID
                        = $this->getSchemaId($permissionSchemaGranter);

                    // Load ungranter
                    $permissionSchemaUngranterID
                        = $this->getSchemaId($permissionSchemaUngranter);

                    // Set schema & permission
                    if (! $invert) {
                        // Grant
                        $this->setSchema(
                            $role['id'],
                            $permissionSchemaGranterID,
                            $permissionSchemaUngranterID
                        );

                        Permission::createPermission($role['id'], $data);
                    } else {
                        // Ungrant
                        $this->setSchema(
                            $role['id'],
                            $permissionSchemaUngranterID,
                            $permissionSchemaGranterID
                        );

                        Permission::deletePermission($role['id'], $data);
                    }
                }

                // Permission has change, rebuild cache
                $content = call_user_func($class.'::find', $uuid);
                if ($content) {
                    $content->touch();
                }
            }

            return $response;
        }

        // Check path and user
        if (! strpos($path, 'api/v1/form/') == 0 || ! $user) {
            return $response;
        }

        preg_match('/^api\/v1\/form\/(.*)?$/', $path, $match);

        if (! isset($match[1])) {
            return $response;
        }

        $exploded = explode('/', $match[1]);
        $uuid = null;

        if (isset($exploded[1])) {
            $match[1] = $exploded[0];
            $uuid = $exploded[1];
        }

        if ($match[1] == 'groups') {
            return $response;
        }

        if (! Schema::hasModel($match[1])) {
            return $response;
        }

        $class = type_to_class($match[1]);
        $content = new $class();
        $action = 'create';

        if (! $this->hasRoleField($class)) {
            return $response;
        }

        // Update action
        if ($uuid) {
            $action = 'update';
            $content = call_user_func($class.'::findOrFail', $uuid);
        }

        // Check permission or super admin
        if (! Permission::allowed($action, $match[1], $uuid)) {
            throw new AuthorizationException();
        }

        return $this->addRoleFieldToForm($response, $content);
    }

    /**
     * @throws AuthorizationException
     */
    protected function handleV2(Request $request, Closure $next): mixed
    {
        $method = $request->getMethod();
        $path = $request->path();
        $roles = null;
        $content = null;

        if (
            strpos($path, 'api/v1/content/') == 0
            && in_array($method, ['POST', 'PUT'])
            && $request->has('roles')
        ) {
            $roles = $request->get('roles');
            unset($request['roles']);
        }

        $response = $next($request);

        if ($response instanceof Response && $response->getStatusCode() != 200) {
            return $response;
        }

        $user = Auth::user();

        // Save roles
        if (str_starts_with($path, 'api/v1/content/') && in_array($method, ['POST', 'PUT'])) {
            /** @var array|null $content */
            $content = json_decode_response($response);

            if ($content === null || ! isset($content['uuid'])) {
                return $response;
            }

            /** @var Content $class */
            $class = type_to_class($request->get('bundle') ?? $content['content_type']);
            /** @var Content|null $content */
            $content = $class::query()->find($content['uuid']);

            if (! $class::isPermissible() || $class::isCategorizable() || ! $content) {
                return $response;
            }

            if ($roles === null) {
                PermissionSchema::resetContentSpecificPermissionsV2($content);
            }
        }

        if ($roles) {
            preg_match('/^api\/v1\/content\/(.*)?$/', $path, $match);

            if (! isset($match[1]) || is_null($content) || ! $content::isPermissible() || $content::isCategorizable()) {
                return $response;
            }

            $roles = collect($roles)->pluck('value', 'id')->filter()->keys();
            $content->contentSpecificPrivileges->roles()->sync($roles);

            return $response;
        }

        // Check path and user
        if (! strpos($path, 'api/v1/form/') == 0 || ! $user) {
            return $response;
        }

        preg_match('/^api\/v1\/form\/(.*)?$/', $path, $match);

        if (! isset($match[1])) {
            return $response;
        }

        $exploded = explode('/', $match[1]);
        $uuid = null;

        if (isset($exploded[1])) {
            $match[1] = $exploded[0];
            $uuid = $exploded[1];
        }

        if ($match[1] == 'groups') {
            return $response;
        }

        if (! Schema::hasModel($match[1])) {
            return $response;
        }

        /** @var Content $class */
        $class = type_to_class($match[1]);
        $content = new $class();
        $action = 'create';

        if (! $class::isPermissible() || $class::isCategorizable()) {
            return $response;
        }

        // Update action
        if ($uuid) {
            $action = 'update';
            $content = call_user_func($class.'::findOrFail', $uuid);
        }

        // Check permission or super admin
        if (! Permission::allowed($action, $match[1], $uuid)) {
            throw new AuthorizationException();
        }

        return $this->addRoleFieldToForm($response, $content);
    }

    /**
     * Return if we need put role on it
     *
     * @param string $class
     *
     * @return bool
     */
    protected function hasRoleField(string $class): bool
    {
        if (! Permission::isPermissible($class)) {
            return false;
        }

        // Try custom callback
        $customHasRole = config('permission.has_role', []);

        foreach ($customHasRole as $callback) {
            if ($callback && is_callable($callback)) {
                $result = $callback(class_to_type($class));
                if (is_bool($result)) {
                    return $result;
                }
            }
        }

        return true;
    }

    /**
     * Add role field to form
     *
     * @param Response $response
     * @param Content $content
     * @return Response
     */
    public function addRoleFieldToForm(Response $response, Content $content)
    {
        /** @var array|null $data */
        $data = json_decode_response($response);
        if ($data === null) {
            return $response;
        }

        $options = InsideSchema::getModelOptions(class_to_type($content));

        if (! array_key_exists('permissible', $options)
            || ! $options['permissible']
            || in_array(
                class_to_type($content),
                config('permission.hidden_types', [])
            )) {
            return $response;
        }
        $tree = RoleService::tree();

        if (Permission::isSystemV2Enabled()) {
            $hasSpecific = $content->uuid && PermissionSchema::hasContentSpecificPermissionV2($content);

            $tree = $this->alterRoleTreeV2($tree, $content, PermissionSchema::getContentSpecificPermissions($content));
        } else {
            $tree = $this->alterRoleTree($tree, $content);
            $hasSpecific = false;

            if ($content->uuid) {
                $hasSpecific = PermissionSchema::hasContentSpecificPermission($content);
            }
        }

        $data['data'][] = [
            'id' => 'group_permissions',
            'weight' => 100,
            'type' => 'fieldset',
            'classes' => '',
            'label' => ['en' => 'Permissions', 'fr' => 'Permissions'],
            'fields' => [
                [
                    'name' => 'roles',
                    'options' => ['tree' => $tree],
                    'type' => 'roles',
                    'weight' => 0,
                    'has_specific' => $hasSpecific,
                ],
            ],
        ];
        set_response($response, $data);

        return $response;
    }

    /**
     * Add can read to all role in tree
     *
     * @param array $tree
     * @param mixed $content
     *
     * @return array
     */
    protected function alterRoleTree(array $tree, $content): array
    {
        foreach ($tree as $key => &$node) {
            if ($node['name'] == 'super_administrator') {
                unset($tree[$key]);
            }

            if ($node['type'] == 'role_category') {
                $node['children'] = $this->alterRoleTree(
                    $node['children'],
                    $content
                );
                continue;
            }

            $node['permission']['read'] = RoleService::can(
                'read',
                $content,
                $node['id']
            );
        }

        // Because unset doesn't reset array keys
        return array_values($tree);
    }

    /**
     * Add can read to all role in tree
     *
     * @param array $tree
     * @param Content $content
     * @param Collection $specificPermissions
     *
     * @return array
     */
    protected function alterRoleTreeV2(array $tree, Content $content, Collection $specificPermissions): array
    {
        foreach ($tree as $key => &$node) {
            if ($node['name'] == 'super_administrator') {
                unset($tree[$key]);
            }

            if ($node['type'] == 'role_category') {
                if (empty($node['children'])) {
                    unset($tree[$key]);
                }

                $node['children'] = $this->alterRoleTreeV2(
                    $node['children'],
                    $content,
                    $specificPermissions
                );
                continue;
            }

            $node['permission']['read'] = $specificPermissions->contains($node['id']);
        }

        // Because unset doesn't reset array keys
        return array_values($tree);
    }

    /**
     * Set wanted schema, delete ungrantid schema for roleId then insiert
     * grantid for roleId
     *
     * @param int $roleId
     * @param int $toGrant
     * @param int $toUngrant
     */
    protected function setSchema(int $roleId, int $toGrant, int $toUngrant): void
    {
        // Ungrant
        try {
            DB::table('inside_roles_permissions_schema')->where([
                'role_id' => $roleId,
                'permission_schema_id' => $toUngrant,
                'is_content_specific' => true,
            ])->delete();
        } catch (QueryException $e) {
            // Do nothing
        }

        // Grant
        try {
            DB::table('inside_roles_permissions_schema')->insert([
                'role_id' => $roleId,
                'permission_schema_id' => $toGrant,
                'is_content_specific' => true,
            ]);
        } catch (QueryException $e) {
            // Do nothing
        }
    }

    protected function getSchemaId(array $schema): int
    {
        $permissionSchema = DB::table('inside_permissions_schema')->where($schema)->first();

        if (isset($permissionSchema->id)) {
            return $permissionSchema->id;
        }

        return DB::table('inside_permissions_schema')->insertGetId($schema);
    }
}
