<?php

namespace Inside\Permission\Http\Controllers;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Inside\Permission\Facades\Role;
use Laravel\Lumen\Routing\Controller;

/**
 * Role Controller
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class RoleController extends Controller
{
    /**
     * List role tree
     */
    public function index(?int $root = null): JsonResponse
    {
        $tree = Role::tree($root);

        return response()->json([
            'data' => $tree,
        ]);
    }

    /**
     * Create a new role
     */
    public function post(Request $request): JsonResponse
    {
        $data = $request->only(
            'name',
            'pid',
            'role_category_id',
            'is_automatic',
            'condition',
            'weight'
        );
        $role = Role::create($data);

        return response()->json([
            'data' => $role->toArray(),
        ]);
    }

    /**
     * Update a role
     */
    public function put(Request $request, int $roleId): JsonResponse
    {
        $data = $request->only(
            'name',
            'pid',
            'role_category_id',
            'is_automatic',
            'condition',
            'weight'
        );
        $role = Role::update($roleId, $data);

        return response()->json([
            'data' => $role->toArray(),
        ]);
    }

    /**
     * Delete a role
     */
    public function delete(int $roleId): JsonResponse
    {
        Role::delete($roleId);

        $tree = Role::tree();

        return response()->json([
            'data' => $tree,
        ]);
    }

    /**
     * List role categories
     */
    public function listCategory(): JsonResponse
    {
        $tree = Role::tree();

        $tree = array_filter(
            $tree,
            function ($value) {
                return $value['type'] == 'role_category';
            }
        );

        return response()->json([
            'data' => $tree,
        ]);
    }

    /**
     * Create a new role
     */
    public function postCategory(Request $request): JsonResponse
    {
        $data = $request->only(
            'name',
            'pid',
            'weight'
        );
        $category = Role::createCategory($data);

        return response()->json([
            'data' => $category,
        ]);
    }

    /**
     * Update a role category
     */
    public function putCategory(Request $request, int $categoryId): JsonResponse
    {
        $data = $request->only(
            'name',
            'pid',
            'weight'
        );
        $category = Role::updateCategory($categoryId, $data);

        return response()->json([
            'data' => $category,
        ]);
    }

    /**
     * Delete a role category
     */
    public function deleteCategory(int $categoryId): JsonResponse
    {
        Role::deleteCategory($categoryId);

        $tree = Role::tree();

        return response()->json([
            'data' => $tree,
        ]);
    }

    /**
     * List user attached to specific role
     */
    public function userList(Request $request, string $roleIds): JsonResponse
    {
        $limit = $request->get('limit');
        $search = $request->get('search');
        $inRole = $request->get('in_role', false);
        $page = $request->get('page', 1);
        $excludeCurrentUser = $request->get('excludeCurrentUser', 0);
        $filters = json_decode($request->get('filters', ''), true);

        return response()->json(Role::listRoleUsers($roleIds, $limit, $search, $inRole, $page, $excludeCurrentUser, $filters));
    }

    /**
     * Assign user attached to specific role
     */
    public function userSync(Request $request, int $roleId): JsonResponse
    {
        $limit = $request->get('limit');

        $assign = (array) $request->get('assign');
        $revoke = (array) $request->get('revoke');

        Role::assignUsersToRole($roleId, $assign);
        Role::revokeUsersToRole($roleId, $revoke);

        $users = Role::listRoleUsers($roleId, $limit);

        return response()->json([
            'data' => $users,
        ]);
    }

    public function userRoles(Request $request): JsonResponse
    {
        $roles = $request->user()->roles?->toArray() ?? [];

        return response()->json([
            'data' => $roles,
        ]);
    }
}
