<?php

namespace Inside\Permission\Exodus\Services;

use function GuzzleHttp\Promise\unwrap;
use Illuminate\Database\Eloquent\Builder as EloquentBuilder;
use Illuminate\Database\Query\Builder as QueryBuilder;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;
use Inside\Content\Facades\ContentCache;
use Inside\Content\Models\Content;
use Inside\Permission\Exodus\Dto\Privileges\BackofficePrivilegeDto;
use Inside\Permission\Exodus\Dto\Privileges\ContentPrivilegeDto;
use Inside\Permission\Exodus\Dto\Privileges\ContentTypePrivilegeDto;
use Inside\Permission\Exodus\Dto\Privileges\MenuPrivilegeDto;
use Inside\Permission\Exodus\Enums\CapabilityEnum;
use Inside\Permission\Exodus\Models\Capability;
use Inside\Permission\Exodus\Models\Privileges\BackofficePrivilege;
use Inside\Permission\Exodus\Models\Privileges\CategorizableContentPrivilege;
use Inside\Permission\Exodus\Models\Privileges\ContentPrivilege;
use Inside\Permission\Exodus\Models\Privileges\ContentSpecificPrivilege;
use Inside\Permission\Exodus\Models\Privileges\ContentTypePrivilege;
use Inside\Permission\Exodus\Models\Privileges\MenuPrivilege;
use Inside\Permission\Exodus\Models\Role;
use Inside\Permission\Exodus\Services\AccessRestriction\BackofficeAccessRestriction;
use Inside\Permission\Exodus\Services\AccessRestriction\CategorizableContentAccessRestriction;
use Inside\Permission\Exodus\Services\AccessRestriction\ContentTypeAccessRestriction;
use Inside\Permission\Exodus\Services\AccessRestriction\MenuAccessRestriction;

final class RolePrivilegesService
{
    /**
     * @var Collection<Role>
     */
    private Collection $roles;

    private string $cacheRolesKey;

    const CACHE_KEY = 'role_privileges_service:_';

    private function __construct(array $roles)
    {
        $this->roles = collect($roles);

        $this->cacheRolesKey = $this->roles->pluck('id')->sort()->implode('_');
    }

    private function cacheKey(): string
    {
        return self::CACHE_KEY.$this->cacheRolesKey;
    }

    public static function getAuthenticated(): Role
    {
        return Role::firstWhere('name', Role::AUTHENTICATED);
    }

    public static function of(Role ...$roles): self
    {
        // Remove generic roles with type like group and workflow
        $roles = collect($roles)->filter(fn (Role $role) => is_null($role->type));

        if (! $roles->contains('name', self::getAuthenticated()->name)) {
            $roles->push(self::getAuthenticated());
        }

        return new self(Collection::unwrap($roles));
    }

    public static function clearAllCache(): void
    {
        collect(Cache::getRedis()->keys('*'.self::CACHE_KEY.'*'))
            ->each(fn (string $key) => Cache::forget(str($key)->after(':')));
    }

    public function clearCache(string $identifier): void
    {
        ContentCache::forget([config('app.key', 'sid2'), 'all']);

        collect(Cache::getRedis()->keys('*'.self::CACHE_KEY.'*'))
            ->filter(fn (string $key) => str_contains($key, $identifier) && str_contains($key, '_'.$this->cacheRolesKey.'_'))
            ->each(fn (string $key) => Cache::forget(str($key)->after(':')));
    }

    /**
     * @return Collection<BackofficePrivilegeDto>
     */
    public function getBackofficePrivileges(): Collection
    {
        $privileges = Cache::rememberForever($this->cacheKey().'_:backoffice', function () {
            return $this->roles
                ->map(fn (Role $role) => $role->backofficePrivileges()->getQuery())
                ->reduce(fn (?EloquentBuilder $query, EloquentBuilder $next) => is_null($query) ? $next : $query->union($next))
                ->get()
                ->map(fn (BackofficePrivilege $backofficeSection) => BackofficePrivilegeDto::fromModel($backofficeSection));
        });

        return BackofficeAccessRestriction::availables()->map(
            fn (BackofficePrivilegeDto $dto) => $dto->setAuthorization($privileges->contains($dto))
        );
    }

    /**
     * @return Collection<int>
     */
    public static function getAuthenticatedBackofficePrivilegesId(): Collection
    {
        return static::getAuthenticated()
            ->getAccessRestriction()
            ->getBackofficePrivileges()
            ->filter(fn (BackofficePrivilegeDto $dto) => $dto->isAuthorized())
            ->map(fn (BackofficePrivilegeDto $dto) => $dto->getId())
            ->values();
    }

    /**
     * @return Collection<ContentTypePrivilegeDto>
     */
    public function getContentTypePrivileges(): Collection
    {
        $privileges = Cache::rememberForever($this->cacheKey().'_:content_type', function () {
            return $this->roles
                ->map(fn (Role $role) => $role->contentTypePrivileges()->getQuery())
                ->reduce(fn (?EloquentBuilder $query, EloquentBuilder $next) => is_null($query) ? $next : $query->union($next))
                ->get()
                ->map(fn (ContentTypePrivilege $contentTypePrivilege) => ContentTypePrivilegeDto::fromModel($contentTypePrivilege));
        });

        return ContentTypeAccessRestriction::availables()->map(
            fn (ContentTypePrivilegeDto $dto) => $dto->setAuthorization($privileges->contains($dto))
        );
    }

    /**
     * @return Collection<int>
     */
    public static function getAuthenticatedContentTypePrivilegesId(): Collection
    {
        return static::getAuthenticated()
            ->getAccessRestriction()
            ->getContentTypePrivileges()
            ->filter(fn (ContentTypePrivilegeDto $dto) => $dto->isAuthorized())
            ->map(fn (ContentTypePrivilegeDto $dto) => $dto->getId())
            ->values();
    }

    /**
     * @return Collection<ContentPrivilegeDto>
     */
    public function getCategorizableContentPrivileges(): Collection
    {
        $privileges = Cache::rememberForever($this->cacheKey().'_:categorizable_content', function () {
            return $this->roles
                ->map(fn (Role $role) => $role->categorizableContentPrivileges()->getQuery())
                ->reduce(fn (?EloquentBuilder $query, EloquentBuilder $next) => is_null($query) ? $next : $query->union($next))
                ->get()
                ->map(fn (CategorizableContentPrivilege $categorizableContentPrivilege) => ContentPrivilegeDto::fromModel($categorizableContentPrivilege));
        });

        return CategorizableContentAccessRestriction::availables()->map(
            fn (ContentPrivilegeDto $dto) => $dto->setAuthorization($privileges->contains($dto))
        );
    }

    /**
     * @return Collection<int>
     */
    public static function getAuthenticatedCategorizableContentPrivilegesId(): Collection
    {
        return static::getAuthenticated()
            ->getAccessRestriction()
            ->getCategorizableContentPrivileges()
            ->filter(fn (ContentPrivilegeDto $dto) => $dto->isAuthorized())
            ->map(fn (ContentPrivilegeDto $dto) => $dto->getId())
            ->values();
    }

    /**
     * @param string|null $langcode
     * @return Collection<MenuPrivilegeDto>
     */
    public function getMenuPrivileges(?string $langcode = null): Collection
    {
        $privileges = Cache::rememberForever($this->cacheKey().'_:menu', function () use ($langcode) {
            return $this->roles
                ->map(fn (Role $role) => $role
                    ->menuPrivileges()
                    ->when($langcode, fn (EloquentBuilder $query, string $langcode) => $query->langcode($langcode))
                    ->getQuery()
                )->reduce(fn (?EloquentBuilder $query, EloquentBuilder $next) => is_null($query) ? $next : $query->union($next))
                ->get()
                ->map(fn (MenuPrivilege $menuPrivilege) => MenuPrivilegeDto::fromModel($menuPrivilege));
        });

        return MenuAccessRestriction::availables()->map(
            fn (MenuPrivilegeDto $dto) => $dto->setAuthorization($privileges->contains($dto))
        );
    }

    /**
     * @return Collection<int>
     */
    public static function getAuthenticatedMenuPrivilegesId(): Collection
    {
        return static::getAuthenticated()
            ->getAccessRestriction()
            ->getMenuPrivileges()
            ->filter(fn (MenuPrivilegeDto $dto) => $dto->isAuthorized())
            ->map(fn (MenuPrivilegeDto $dto) => $dto->getId())
            ->values();
    }

    public function applyCategorizableContentAccessRestriction(QueryBuilder|EloquentBuilder $query, array $capabilities = [CapabilityEnum::READ]): void
    {
        $query->whereIn('uuid_host', fn (QueryBuilder|EloquentBuilder $query) => $query
            ->select('uuid_host')
            ->from(CategorizableContentPrivilege::TABLE)
            ->join(CategorizableContentPrivilege::PIVOT_ROLE_TABLE, CategorizableContentPrivilege::PIVOT_ROLE_TABLE.'.content_privilege_id', '=', CategorizableContentPrivilege::TABLE.'.id')
            ->whereIn('capability_id', fn (QueryBuilder|EloquentBuilder $query) => $query->from(Capability::TABLE)->whereIn('name', $capabilities)->select('id'))
            ->whereIn('role_id', $this->roles->pluck('id'))
        );
    }

    private function applyContentAccessRestriction(QueryBuilder|EloquentBuilder $query, Content $content, array $capabilities = [CapabilityEnum::READ], string $boolean = 'and', bool $false = false): void
    {
        $query->whereIn($content->getTable().'.uuid', fn (QueryBuilder|EloquentBuilder $query) => $query
            ->select('uuid')
            ->from(ContentPrivilege::TABLE)
            ->join(ContentPrivilege::PIVOT_ROLE_TABLE, ContentPrivilege::PIVOT_ROLE_TABLE.'.content_privilege_id', '=', ContentPrivilege::TABLE.'.id')
            ->whereIn('capability_id', fn (QueryBuilder|EloquentBuilder $query) => $query->from(Capability::TABLE)->whereIn('name', $capabilities)->select('id'))
            ->whereIn('role_id', $this->roles->pluck('id')),
        $boolean, $false);
    }

    private function applyAuthorContentAccessRestriction(QueryBuilder|EloquentBuilder $query, Content $content, string $author): void
    {
        $query->where($content->getTable().'.author_id', $author);
    }

    private function applyContentSpecificExcludedRestriction(QueryBuilder|EloquentBuilder $query, Content $content): void
    {
        $query->whereNotIn($content->getTable().'.uuid', fn (QueryBuilder|EloquentBuilder $query) => $query
            ->select('uuid')
            ->from(ContentSpecificPrivilege::TABLE)
            ->join(ContentSpecificPrivilege::PIVOT_ROLE_TABLE, ContentSpecificPrivilege::PIVOT_ROLE_TABLE.'.content_privilege_id', '=', ContentSpecificPrivilege::TABLE.'.id')
        );
    }

    private function applyContentSpecificIncludedRestriction(QueryBuilder|EloquentBuilder $query, Content $content): void
    {
        $query->whereIn($content->getTable().'.uuid', fn (QueryBuilder|EloquentBuilder $query) => $query
            ->select('uuid')
            ->from(ContentSpecificPrivilege::TABLE)
            ->join(ContentSpecificPrivilege::PIVOT_ROLE_TABLE, ContentSpecificPrivilege::PIVOT_ROLE_TABLE.'.content_privilege_id', '=', ContentSpecificPrivilege::TABLE.'.id')
            ->whereIn('role_id', $this->roles->pluck('id'))
        );
    }

    public function applyContentViewRestriction(QueryBuilder|EloquentBuilder $query, Content $content): void
    {
        $query->where(function (QueryBuilder|EloquentBuilder $query) use ($content) {
            $query->where(function (QueryBuilder|EloquentBuilder $query) use ($content) {
                $this->applyContentSpecificExcludedRestriction($query, $content);
                $this->applyContentAccessRestriction($query, $content);
            })->orWhere(function (QueryBuilder|EloquentBuilder $query) use ($content) {
                $this->applyContentSpecificIncludedRestriction($query, $content);
            })->orWhere(function (QueryBuilder|EloquentBuilder $query) use ($content) {
                $this->applyAuthorContentAccessRestriction($query, $content, Auth::user()->uuid);
            });
        });
    }

    public function applyContentUpdateRestriction(QueryBuilder|EloquentBuilder $query, Content $content): void
    {
        $query->where(function (QueryBuilder|EloquentBuilder $query) use ($content) {
            $query->where(function (QueryBuilder|EloquentBuilder $query) use ($content) {
                $this->applyContentSpecificExcludedRestriction($query, $content);
                $this->applyContentAccessRestriction($query, $content, [CapabilityEnum::READ, CapabilityEnum::UPDATE]);
            })->orWhere(function (QueryBuilder|EloquentBuilder $query) use ($content) {
                $this->applyContentSpecificIncludedRestriction($query, $content);
            });
        });
    }

    public function applyContentDeleteRestriction(QueryBuilder|EloquentBuilder $query, Content $content): void
    {
        $query->where(function (QueryBuilder|EloquentBuilder $query) use ($content) {
            $query->where(function (QueryBuilder|EloquentBuilder $query) use ($content) {
                $this->applyContentSpecificExcludedRestriction($query, $content);
                $this->applyContentAccessRestriction($query, $content, [CapabilityEnum::DELETE]);
            })->orWhere(function (QueryBuilder|EloquentBuilder $query) use ($content) {
                $this->applyContentSpecificIncludedRestriction($query, $content);
            });
        });
    }

    private function canAccessContent(Content $content, string $capability): bool
    {
        /**
         * If the content does not have a uuid, we will check the general privileges for the content type
         */
        if (is_null($content->uuid)) {
            return ContentTypePrivilege::query()
                ->whereHas('roles', fn (EloquentBuilder $query) => $query->whereIn('role_id', $this->roles->pluck('id')))
                ->whereHas('capability', fn (EloquentBuilder $query) => $query->where('name', $capability))
                ->where('type', $content::class)
                ->exists();
        }

        /**
         * If the content get specific privileges, and the capability is READ
         * We need to check if the user has the specific privilege, otherwise we will check the general privilege
         */
        $query = $capability === CapabilityEnum::READ && $content->hasSpecificPrivilege()
            ? ContentSpecificPrivilege::query()
            : ContentPrivilege::query();

        return $query->whereHas('roles', fn (EloquentBuilder $query) => $query->whereIn('role_id', $this->roles->pluck('id')))
            ->whereHas('capability', fn (EloquentBuilder $query) => $query->where('name', $capability))
            ->where('uuid', $content->uuid)
            ->exists();
    }

    private function canAccessCategorizableContent(Content $content, string $capability): bool
    {
        return CategorizableContentPrivilege::query()
            ->whereHas('roles', fn (EloquentBuilder $query) => $query->whereIn('role_id', $this->roles->pluck('id')))
            ->whereHas('capability', fn (EloquentBuilder $query) => $query->where('name', $capability))
            ->where('uuid_host', $content->uuid_host)
            ->exists();
    }

    public function canUpdateContent(Content $content): bool
    {
        return $this->canAccessContent($content, CapabilityEnum::UPDATE);
    }

    public function canReadContent(Content $content): bool
    {
        return $this->canAccessContent($content, CapabilityEnum::READ);
    }

    public function canDeleteContent(Content $content): bool
    {
        return $this->canAccessContent($content, CapabilityEnum::DELETE);
    }

    public function canReadCategorizableContent(Content $content): bool
    {
        return $this->canAccessCategorizableContent($content, CapabilityEnum::READ);
    }

    public function canUpdateCategorizableContent(Content $content): bool
    {
        return $this->canAccessCategorizableContent($content, CapabilityEnum::UPDATE);
    }

    public function canAssignCategorizableContent(Content $content): bool
    {
        return $this->canAccessCategorizableContent($content, CapabilityEnum::ASSIGN);
    }

    public function canDeleteCategorizableContent(Content $content): bool
    {
        return $this->canAccessCategorizableContent($content, CapabilityEnum::DELETE);
    }
}
