<?php

namespace Inside\Permission\Exodus\Services\ComputeRestriction;

use Illuminate\Database\Query\Builder;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Content;
use Inside\Host\Exodus\Services\ContentTypeStatusService;
use Inside\Permission\Exodus\Dto\Privileges\ContentPrivilegeDto;
use Inside\Permission\Exodus\Dto\Privileges\ContentTypePrivilegeDto;
use Inside\Permission\Exodus\Enums\CapabilityEnum;
use Inside\Permission\Exodus\Models\Capability;
use Inside\Permission\Exodus\Models\Privileges\CategorizableContentPrivilege;
use Inside\Permission\Exodus\Models\Privileges\ContentPrivilege;
use Inside\Permission\Exodus\Models\Role;
use Inside\Permission\Exodus\Models\ViewModels\CategorizableContentIndex;
use Inside\Permission\Exodus\Models\ViewModels\CategorizableContentTranslationIndexes;
use Inside\Permission\Exodus\Models\ViewModels\ContentTranslationIndexes;
use Inside\Permission\Exodus\Services\RolePrivilegesService;

class ComputeRestrictionService
{
    public static function getCategorizableContentTypes(): Collection
    {
        /** @var Collection $categorizables */
        $categorizables = app(ContentTypeStatusService::class)
            ->permissibles()
            ->categorizables()
            ->keys()
            ->map(fn (string $type) => type_to_class($type));

        return $categorizables;
    }

    public static function getPermissibleButNotCategorizableContentTypes(): Collection
    {
        /** @var Collection $contentTypes */
        $contentTypes = app(ContentTypeStatusService::class)
            ->permissibles()
            ->notCategorizables()
            ->keys()
            ->map(fn (string $type) => type_to_class($type));

        return $contentTypes;
    }

    public static function getContentTypesFor(Role $role): Collection
    {
        return $role->getAccessRestriction()
            ->getContentTypePrivileges()
            ->filter(fn (ContentTypePrivilegeDto $dto) => $dto->isAuthorized())
            ->map(fn ($contentType) => $contentType->toArray());
    }

    public static function getCategorizableContentsFor(Role $role): Collection
    {
        return $role->getAccessRestriction()
            ->getCategorizableContentPrivileges()
            ->filter(fn (ContentPrivilegeDto $dto) => $dto->isAuthorized())
            ->map(fn ($categorizable) => $categorizable->toArray());
    }

    public static function retrieveExistingContentPrivilegesFor(string $capability): Collection
    {
        return DB::table(ContentPrivilege::TABLE)
            ->where('capability_id', '=', fn (Builder $query) => $query->from(Capability::TABLE)->where('name', '=', $capability)->select('id'))
            ->leftJoin(ContentTranslationIndexes::TABLE, ContentPrivilege::TABLE.'.uuid', '=', ContentTranslationIndexes::TABLE.'.translatable_uuid')
            ->leftJoin('inside_pivots', ContentPrivilege::TABLE.'.uuid', '=', 'inside_pivots.parent_uuid')
            ->select([
                ContentPrivilege::TABLE.'.id',
                ContentTranslationIndexes::TABLE.'.translatable_uuid',
                'related_uuid',
                ContentTranslationIndexes::TABLE.'.translatable_type',
                'related_type',
            ])
            ->get();
    }

    // TODO: Add a description to explain
    public static function filterPrivilegesFor(Collection $privileges, Collection $categorizableContentTypes, Collection $notCategorizablesContentTypes, Collection $viewableContentTypes): Collection
    {
        return $privileges
            ->map(fn ($content) => [
                'privilege_id' => $content->id,
                'uuid' => $content->translatable_uuid,
                'type' => $content->translatable_type,
                'categorizable_uuid' => $content->related_uuid,
                'categorizable_type' => $content->related_type,
            ])
            ->filter(function ($content) use ($categorizableContentTypes, $viewableContentTypes, $notCategorizablesContentTypes) {
                return $viewableContentTypes->contains($content['type']) && (
                    is_null($content['categorizable_type']) ||
                    $categorizableContentTypes->contains($content['categorizable_type']) ||
                    $notCategorizablesContentTypes->contains($content['categorizable_type'])
                );
            })
            ->groupBy('uuid')
            ->map(function (Collection $contents) use ($categorizableContentTypes, $notCategorizablesContentTypes) {
                $categorizables = $contents
                    ->whereIn('categorizable_type', $categorizableContentTypes)
                    ->mapToGroups(fn ($content) => [$content['categorizable_type'] => $content['categorizable_uuid']])
                    ->all();

                $notCategorizables = $contents
                    ->whereIn('categorizable_type', $notCategorizablesContentTypes)
                    ->map(fn ($content) => $content['categorizable_type'])
                    ->unique()
                    ->all();

                return [
                    'privilege_id' => $contents->first()['privilege_id'],
                    'type' => $contents->first()['type'],
                    'categorizables' => $categorizables,
                    'not_categorizables' => $notCategorizables,
                ];
            });
    }

    public static function precomputeCapabilitiesOnContentForRoles(Content $content, array $restrictions)
    {
        // Now we need to find roles that successfully match the requirements
        // For the viewable privilege, we need to get access at least to one of the categorizable contents for each type
        // For the other privileges, we need to get access to all of the categorizable contents for each type
        $compute = DB::table(CategorizableContentPrivilege::TABLE.' AS privileges')
            ->join(CategorizableContentPrivilege::PIVOT_ROLE_TABLE.' AS privilege_role', 'privileges.id', 'privilege_role.content_privilege_id')
            ->join(CategorizableContentIndex::TABLE.' AS indexes', 'privileges.uuid_host', 'indexes.uuid_host')
            ->join(Capability::TABLE.' AS capabilities', 'privileges.capability_id', 'capabilities.id')
            ->select([
                'privilege_role.role_id',
                'privileges.capability_id',
                'capabilities.name',
                'indexes.type',
                DB::raw('COUNT(DISTINCT indexes.uuid_host) AS count'),
            ])
            ->whereIn('capabilities.name', [CapabilityEnum::READ, CapabilityEnum::ASSIGN])
            ->where(function ($query) use ($restrictions) {
                foreach ($restrictions['categorizables'] as $class => $uuid_hosts) {
                    $query->orWhere(function ($subQuery) use ($class, $uuid_hosts) {
                        $subQuery->where('indexes.type', $class)->whereIn('indexes.uuid_host', $uuid_hosts);
                    });
                }
            })
            ->groupBy('privilege_role.role_id', 'privileges.capability_id', 'capabilities.name', 'indexes.type')
            ->orderBy('indexes.type')
            ->get();

        /** @var Collection<Role> $roles */
        $roles = Role::query()
            ->whereIn('id', $compute->pluck('role_id')->unique())
            ->where('name', '!=', Role::SUPER_ADMINISTRATOR)
            ->get();

        $precomputed = [];

        foreach ($roles as $role) {
            $categorizableDependencies = $compute
                ->where('role_id', $role->id)
                ->whereIn('type', $restrictions['categorizables'])
                ->reject(fn ($privilege) => $privilege->name !== CapabilityEnum::READ &&
                    $privilege->count < count($restrictions['categorizables'][$privilege->type] ?? [])
                )->pluck('name')->unique()->toArray();

            $contentDependencies = self::getContentTypesFor($role)
                ->where('capability.name', CapabilityEnum::READ)
                ->whereIn('type', $restrictions['content_types'])
                ->where('is_authorized', true)
                ->count();

            $hasContentRequirements = $contentDependencies === count($restrictions['content_types']);

            if (empty($categorizableDependencies) && ! $hasContentRequirements) {
                continue;
            }

            $allowed = [];

            if (empty($categorizableDependencies) || in_array(CapabilityEnum::ASSIGN, $categorizableDependencies)) {
                if ($hasContentRequirements && $role->hasContentTypePrivilegeTo(CapabilityEnum::UPDATE, $content)) {
                    $allowed[] = CapabilityEnum::UPDATE;
                }

                if ($hasContentRequirements && $role->hasContentTypePrivilegeTo(CapabilityEnum::DELETE, $content)) {
                    $allowed[] = CapabilityEnum::DELETE;
                }

                if ($hasContentRequirements && $content::isCategorizable() && $role->hasContentTypePrivilegeTo(CapabilityEnum::CREATE, $content)) {
                    $allowed[] = CapabilityEnum::ASSIGN;
                }
            }

            if ($hasContentRequirements && (empty($categorizableDependencies) || in_array(CapabilityEnum::READ, $categorizableDependencies))) {
                $allowed[] = CapabilityEnum::READ;
            }

            $precomputed[] = [
                'role' => [
                    'id' => $role->id,
                    'name' => $role->name,
                    'is_super_admin' => $role->isSuperAdmin(),
                    'is_authenticated' => $role->isAuthenticated(),
                ],
                'allowed' => $allowed,
            ];
        }

        return $precomputed;
    }

    public static function computeCapabilitiesOnContentForRoles(Content $content): void
    {
        $precomputed = self::precomputeCapabilitiesOnContentForRoles($content, $content->getRestrictionRequirements());

        // We need to get the privileges for this content
        $privileges = $content->getPrivileges();

        $isCategorizable = $content::isCategorizable();

        foreach ($precomputed as $details) {
            $pivot = array_values(Arr::only($privileges, $details['allowed']));

            if ($isCategorizable) {
                Role::find($details['role']['id'])?->categorizableContentPrivileges()->syncWithoutDetaching($pivot);
            } else {
                Role::find($details['role']['id'])?->contentPrivileges()->syncWithoutDetaching($pivot);
            }
        }

        RolePrivilegesService::clearAllCache();

        Log::info('[PERMISSION] Compute restriction for content saved', [
            'content' => $content->uuid,
            'type' => $content::class,
            'precomputed' => $precomputed,
            'privileges' => $privileges,
        ]);
    }

    /**
     * Role need at least read access when get assign / update or delete access.
     * When Authenticated Role lost the read access, it will be removed from the role due to heritage.
     * So we need to grant the read access to the role if it has assign / update or delete access for each categorizable content.
     *
     * @param Role $role
     * @return void
     */
    public static function computeCategorizableCapabilitiesFor(Role $role): void
    {
        $missing = $role
            ->getAccessRestriction()
            ->getCategorizableContentPrivileges()
            ->map(fn (ContentPrivilegeDto $dto) => [
                'id' => $dto->getId(),
                'capability' => $dto->getCapability()->getName(),
                'uuid_host' => $dto->getIndex()?->getUuidHost(),
                'authorized' => $dto->isAuthorized(),
            ])->groupBy('uuid_host')
            ->reject(function (Collection $content) {
                $capabilities = $content->pluck('authorized', 'capability');

                if ($capabilities->get(CapabilityEnum::ASSIGN) || $capabilities->get(CapabilityEnum::UPDATE) || $capabilities->get(CapabilityEnum::DELETE)) {
                    return $capabilities->get(CapabilityEnum::READ);
                }

                return true;
            })->flatten(1)
            ->where('capability', CapabilityEnum::READ)
            ->pluck('id')
            ->values()
            ->toArray();

        $role->categorizableContentPrivileges()->syncWithoutDetaching($missing);
    }

    public static function computeCapabilitiesFor(Role $role): void
    {
        $categorizablesContentTypes = self::getCategorizableContentTypes();
        $notCategorizablesContentTypes = self::getPermissibleButNotCategorizableContentTypes();

        $contentTypes = self::getContentTypesFor($role);
        $viewableContentTypes = $contentTypes->where('capability.name', CapabilityEnum::READ)->pluck('type')->flatten();
        $categorizablesContents = self::getCategorizableContentsFor($role);

        $viewableContents = self::retrieveExistingContentPrivilegesFor(CapabilityEnum::READ);
        $viewableCategorizableContents = $categorizablesContents->where('capability.name', CapabilityEnum::READ)->pluck('index.translations.*.uuid')->flatten();
        $viewConcerns = self::filterPrivilegesFor($viewableContents, $categorizablesContentTypes, $notCategorizablesContentTypes, $viewableContentTypes)
            ->filter(function ($content) use ($viewableCategorizableContents, $viewableContentTypes) {
                return collect($content['categorizables'])->map(fn (Collection $categorizables, string $type) => $viewableCategorizableContents->intersect($categorizables)->isNotEmpty())->reject()->isEmpty()
                    && collect($content['not_categorizables'])->diff($viewableContentTypes)->isEmpty();
            })->pluck('privilege_id');
        unset($viewableContents, $viewableCategorizableContents);

        $updatableContents = self::retrieveExistingContentPrivilegesFor(CapabilityEnum::UPDATE);
        $assignableCategorizableContents = $categorizablesContents->where('capability.name', CapabilityEnum::ASSIGN)->pluck('index.translations.*.uuid')->flatten();
        $updatableContentTypes = $contentTypes->where('capability.name', CapabilityEnum::UPDATE)->pluck('type')->flatten();
        $updateConcerns = self::filterPrivilegesFor($updatableContents, $categorizablesContentTypes, $notCategorizablesContentTypes, $updatableContentTypes)
            ->filter(function ($content) use ($assignableCategorizableContents, $viewableContentTypes) {
                return collect($content['categorizables'])->map(fn (Collection $categorizables, string $type) => $categorizables->diff($assignableCategorizableContents)->isEmpty())->reject()->isEmpty()
                    && collect($content['not_categorizables'])->diff($viewableContentTypes)->isEmpty();
            })->pluck('privilege_id');
        unset($updatableContents, $updatableContentTypes);

        $deletableContents = self::retrieveExistingContentPrivilegesFor(CapabilityEnum::DELETE);
        $deletableContentTypes = $contentTypes->where('capability.name', CapabilityEnum::DELETE)->pluck('type')->flatten();
        $deleteConcerns = self::filterPrivilegesFor($deletableContents, $categorizablesContentTypes, $notCategorizablesContentTypes, $deletableContentTypes)
            ->filter(function ($content) use ($assignableCategorizableContents, $viewableContentTypes) {
                return collect($content['categorizables'])->map(fn (Collection $categorizables, string $type) => $categorizables->diff($assignableCategorizableContents)->isEmpty())->reject()->isEmpty()
                    && collect($content['not_categorizables'])->diff($viewableContentTypes)->isEmpty();
            })->pluck('privilege_id');

        unset($deletableContents, $deletableContentTypes, $assignableCategorizableContents);

        $role->contentPrivileges()->sync($viewConcerns->merge($updateConcerns)->merge($deleteConcerns));
    }
}
