<?php

namespace Inside\Permission\Exodus\Models\Traits;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Support\Collection;
use Inside\Content\Facades\Schema;
use Inside\Host\Exodus\Services\ContentTypeStatusService;
use Inside\Permission\Exodus\Models\Privileges\CategorizableContentPrivilege;
use Inside\Permission\Exodus\Models\Privileges\ContentPrivilege;
use Inside\Permission\Exodus\Models\Privileges\ContentSpecificPrivilege;
use Inside\Permission\Exodus\Models\Privileges\ContentTypePrivilege;

trait HasPrivileges
{
    /**
     * @mixin Builder
     * @return MorphMany
     */
    public function categorizablePrivileges(): MorphMany
    {
        return $this->morphMany(CategorizableContentPrivilege::class, 'rightable', 'rightable_type', 'rightable_uuid_host', 'uuid_host');
    }

    /**
     * @return HasMany
     */
    public function contentPrivileges(): HasMany
    {
        return $this->hasMany(ContentPrivilege::class, 'uuid', 'uuid');
    }

    /**
     * @return HasOne
     */
    public function contentSpecificPrivileges(): HasOne
    {
        return $this->hasOne(ContentSpecificPrivilege::class, 'uuid', 'uuid');
    }

    /**
     * @return Builder
     */
    public static function contentTypePrivileges(): Builder
    {
        return ContentTypePrivilege::query()->where('type', static::class);
    }

    public static function getCategorizablesContentType(): Collection
    {
        return app(ContentTypeStatusService::class)->categorizables()->keys();
    }

    public static function getCategorizableRelations(): Collection
    {
        $relations = Schema::getRelationsWithOption(class_to_type(static::class));

        return collect($relations)
            ->reject('authors')
            ->map(fn ($relation) => data_get($relation, 'target.0', null))
            ->intersect(static::getCategorizablesContentType());
    }

    public static function getRequiredCategorizableRelations(): Collection
    {
        $relations = Schema::getRelationsWithOption(class_to_type(static::class));

        return collect($relations)
            ->reject('authors')
            ->where('required', true)
            ->map(fn ($relation) => data_get($relation, 'target.0', null))
            ->intersect(static::getCategorizablesContentType());
    }

    public function getCategorizableContents(): Collection
    {
        return static::getCategorizableRelations()->keys()->mapWithKeys(fn (string $relation) => [
            $relation => $this->{$relation}->where('status', 1)->pluck('uuid_host'),
        ]);
    }
}
