<?php

namespace Inside\Permission\Exodus\Actions\StorePrivileges;

use Illuminate\Support\Collection;
use Inside\Permission\Exodus\Actions\RemoveDuplicateAuthenticatedPrivileges\RemoveDuplicateAuthenticatedPrivilegesMenu;
use Inside\Permission\Exodus\Dto\CapabilityDto;
use Inside\Permission\Exodus\Dto\Indexes\MenuIndexDto;
use Inside\Permission\Exodus\Dto\Privileges\MenuPrivilegeDto;
use Inside\Permission\Exodus\Models\Role;

class StoreMenuPrivileges
{
    public function execute(Role $role, array $data): void
    {
        // Clear inside cache
        $role->getAccessRestriction()->clearCache('menu');

        $privileges = collect($data)
            ->flatten(1)
            ->map(fn (array $menu) => $this->extractPrivileges(collect($menu)))
            ->flatten()
            ->filter(fn (MenuPrivilegeDto $dto) => $dto->isAuthorized())
            ->map(fn (MenuPrivilegeDto $dto) => $dto->getId());

        // Sync privileges
        $role->menuPrivileges()->sync($privileges);

        // Remove duplicate authenticated privileges
        if ($role->name === Role::AUTHENTICATED) {
            (new RemoveDuplicateAuthenticatedPrivilegesMenu())->execute();
        }

        $role->computeRestriction();
    }

    private function extractPrivileges(Collection $menu): array
    {
        $children = collect($menu->pull('children'));

        $current = collect($menu->get('privileges'))
            ->map(fn (array $privilege) => MenuPrivilegeDto::from(
                MenuIndexDto::from(
                    $menu['uuid'],
                    $menu['type']
                ),
                CapabilityDto::from(
                    $privilege['capability'],
                    $privilege['capability_id']
                ),
                $privilege['privilege_id'],
            )->setAuthorization($privilege['is_inherited'] ? false : $privilege['is_authorized']));

        if ($children->isNotEmpty()) {
            $children->transform(fn (array $child) => $this->extractPrivileges(collect($child)));
        }

        return $current->push($children)->flatten()->toArray();
    }
}
