<?php

namespace Inside\Permission\Exodus\Actions\StorePrivileges;

use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Log;
use Inside\Permission\Exodus\Actions\RemoveDuplicateAuthenticatedPrivileges\RemoveDuplicateAuthenticatedPrivilegesCategorizable;
use Inside\Permission\Exodus\Dto\CapabilityDto;
use Inside\Permission\Exodus\Dto\Indexes\PermissibleContentIndexDto;
use Inside\Permission\Exodus\Dto\Privileges\ContentPrivilegeDto;
use Inside\Permission\Exodus\Enums\CapabilityEnum;
use Inside\Permission\Exodus\Http\Resources\CategorizableContentPrivilegeCollection;
use Inside\Permission\Exodus\Jobs\ComputeRoleRestriction;
use Inside\Permission\Exodus\Models\Capability;
use Inside\Permission\Exodus\Models\Privileges\ContentTypePrivilege;
use Inside\Permission\Exodus\Models\Role;
use Inside\Permission\Exodus\Services\RolePrivilegesService;

class StoreCategorizableContentPrivileges
{
    public function execute(Role $role, array $data): void
    {
        // Clear inside cache
        $role->getAccessRestriction()->clearCache('categorizable_content');
        $role->getAccessRestriction()->clearCache('content_type');

        $query = ContentTypePrivilege::query()->select('id');
        $capabilities = Capability::all();

        $privileges = collect($data)
            ->flatten(1)
            ->map(fn (array $content) => $this->extractPrivileges(collect($content)))
            ->flatten()
            ->filter(fn (ContentPrivilegeDto $dto) => $dto->isAuthorized());

        Log::info('[StoreCategorizableContentPrivileges] Storing '.$privileges->count().' '.json_encode($privileges->pluck('id')->toArray())." privileges for role {$role->name} ({$role->id})");

        // Retrieve all content type privileges determinated throught categorizable content privileges
        // If the capability is ASSIGN, we don't need to store it because it's not a content type privilege
        $privileges
            ->map(function (ContentPrivilegeDto $dto) {
                $capability = $dto->getCapability();
                $type = $dto->getIndex()?->getType();

                if ($capability->getName() === CapabilityEnum::ASSIGN) {
                    return null;
                }

                $capability_id = $capability->getId();

                return [
                    ['capability_id', '=', $capability_id],
                    ['type', '=',  $type],
                ];
            })
            ->filter()
            ->unique()
            ->each(fn ($data) => $query->orWhere($data));

        // Get all privileges ids
        $privileges = $privileges->map(fn (ContentPrivilegeDto $dto) => $dto->getId())->values();

        // Get all content type privileges that are not categorizable and merge with the new ones determinated throught categorizable content privileges
        $concerns = $role->contentTypePrivileges
            ->reject(fn (ContentTypePrivilege $privilege) => $privilege->type::isCategorizable())
            ->pluck('id')
            ->merge($query->get()->pluck('id')->values());

        Log::info('[StoreCategorizableContentPrivileges] Storing '.$concerns->count().' '.json_encode($concerns->toArray())." content type privileges for role {$role->name} ({$role->id})");

        // Sync content type privileges
        if (! empty($query->getQuery()->wheres)) {
            $role->contentTypePrivileges()->syncWithoutDetaching($concerns);
        }

        // Sync privileges
        $role->categorizableContentPrivileges()->sync($privileges);

        // Remove duplicate authenticated privileges
        if ($role->name === Role::AUTHENTICATED) {
            (new RemoveDuplicateAuthenticatedPrivilegesCategorizable())->execute();

            Role::withoutDeprecatedRoles()
                ->withoutAuthenticated()
                ->withoutSuperAdministrator()
                ->each(fn (Role $role) => ComputeRoleRestriction::dispatch($role));
        }

        // Recompute restrictions
        Log::info("[ComputeRoleRestrictions] Computing role restrictions for role {$role->name} ({$role->id})");
        $role->computeRestriction();
    }

    private function extractPrivileges(Collection $content): array
    {
        $children = collect($content->pull('children'));

        $current = collect($content->get('privileges'))
            ->map(fn (array $privilege) => ContentPrivilegeDto::from(
                PermissibleContentIndexDto::from(
                    $content['uuid_host'],
                    $content['type'],
                    collect()
                ),
                CapabilityDto::from(
                    $privilege['capability'],
                    $privilege['capability_id']
                ),
                $privilege['privilege_id'],
            )->setAuthorization($privilege['is_inherited'] ? false : $privilege['is_authorized']));

        if ($children->isNotEmpty()) {
            $children->transform(fn (array $child) => $this->extractPrivileges(collect($child)));
        }

        return $current->push($children)->flatten()->toArray();
    }
}
