<?php

namespace Inside\Permission\Exodus\Actions\StorePrivileges;

use Illuminate\Support\Collection;
use Inside\Permission\Exodus\Actions\RemoveDuplicateAuthenticatedPrivileges\RemoveDuplicateAuthenticatedPrivilegesBackofficeSections;
use Inside\Permission\Exodus\Dto\CapabilityDto;
use Inside\Permission\Exodus\Dto\Privileges\BackofficePrivilegeDto;
use Inside\Permission\Exodus\Models\Role;

class StoreBackofficePrivileges
{
    public function execute(Role $role, array $data): void
    {
        // Clear inside cache
        $role->getAccessRestriction()->clearCache('backoffice');

        $privileges = collect($data)
            ->map(fn (array $backoffice) => $this->extractPrivileges(collect($backoffice)))
            ->flatten(1)
            ->filter(fn (BackofficePrivilegeDto $dto) => $dto->isAuthorized())
            ->map(fn (BackofficePrivilegeDto $dto) => $dto->getId())
            ->values();

        // Sync privileges
        $role->backofficePrivileges()->sync($privileges);

        // Remove duplicate authenticated privileges
        if ($role->name === Role::AUTHENTICATED) {
            (new RemoveDuplicateAuthenticatedPrivilegesBackofficeSections())->execute();
        }

        $role->computeRestriction();
    }

    private function extractPrivileges(Collection $backoffice): array
    {
        return collect($backoffice->get('privileges'))
            ->map(fn (array $privilege) => BackofficePrivilegeDto::from(
                $backoffice->get('name'),
                CapabilityDto::from(
                    $privilege['capability'],
                    $privilege['capability_id']
                ),
                $privilege['privilege_id'],
            )->setAuthorization($privilege['is_inherited'] ? false : $privilege['is_authorized']))
            ->toArray();
    }
}
