<?php

namespace Inside\Permission\Exodus\Actions\ConvertOldPermissions;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Inside\Host\Exodus\Services\ContentTypeStatusService;
use Inside\Permission\Exodus\Enums\CapabilityEnum;
use Inside\Permission\Exodus\Models\Capability;
use Inside\Permission\Exodus\Models\Privileges\ContentTypePrivilege;
use Inside\Permission\Exodus\Models\Role;

class ConvertOldContentTypePermissions
{
    public function execute(): void
    {
        $privileges = ContentTypePrivilege::all();
        $contentTypes = app(ContentTypeStatusService::class)->permissibles()->keys()->map(fn ($type) => type_to_class($type));
        $capabilities = Capability::all();
        $roles = Role::where('name', 'not like', 'group-%')->get();

        DB::table('inside_roles_permissions_schema')
            ->join('inside_permissions_schema', 'inside_roles_permissions_schema.permission_schema_id', '=', 'inside_permissions_schema.id')
            ->where('is_content_specific', false)
            ->whereIn('role_id', $roles->pluck('id'))
            ->whereIn('action', ['read', 'create', 'update', 'delete'])
            ->where('authorizable_uuid', null)
            ->where('invert', false)
            ->get()
            ->map(function ($permission) use ($capabilities, $privileges) {
                $action = match ($permission->action) {
                    'create' => $capabilities->firstWhere('name', CapabilityEnum::CREATE)->id,
                    'update' => $capabilities->firstWhere('name', CapabilityEnum::UPDATE)->id,
                    'delete' => $capabilities->firstWhere('name', CapabilityEnum::DELETE)->id,
                    default => $capabilities->firstWhere('name', CapabilityEnum::READ)->id,
                };

                $privilege = $privileges->where('type', $permission->authorizable_type)->where('capability_id', $action)->first()?->id;

                if (is_null($privilege)) {
                    Log::warning("[Inside Permission] Could not find privilege for permission: type: {$permission->type}, action: {$permission->action}, role: {$permission->role_id}");

                    return false;
                }

                return [
                    'role_id' => $permission->role_id,
                    'privilege_id' => $privilege,
                ];
            })
            ->filter()
            ->groupBy('privilege_id')
            ->each(function ($roles, $privilege) use ($privileges) {
                Log::info("[Inside Permission] Syncing roles [{$roles->pluck('role_id')->implode(', ')}] to privilege {$privilege}");

                $privileges->firstWhere('id', $privilege)->roles()->sync($roles->pluck('role_id')->unique());
            });
    }
}
