<?php

namespace Inside\Permission\Exodus\Actions\ConvertOldPermissions;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Inside\Host\Exodus\Services\ContentTypeStatusService;
use Inside\Permission\Exodus\Enums\CapabilityEnum;
use Inside\Permission\Exodus\Models\Capability;
use Inside\Permission\Exodus\Models\Privileges\CategorizableContentPrivilege;
use Inside\Permission\Exodus\Models\Privileges\ContentTypePrivilege;
use Inside\Permission\Exodus\Models\ViewModels\CategorizableContentTranslationIndexes;
use Inside\Permission\Models\Role;

class ConvertOldCategorizableContentPermissions
{
    public function execute(): void
    {
        $roles = Role::where('name', 'not like', 'group-%')->get();
        $contentTypes = app(ContentTypeStatusService::class)->categorizables()->permissibles()->keys()->map(fn ($type) => type_to_class($type));

        Log::info('[Inside Permission] Converting old categorizable content permissions');

        $privileges = DB::table('inside_permissions')
            ->join(CategorizableContentTranslationIndexes::TABLE, 'inside_permissions.uuid', '=', CategorizableContentTranslationIndexes::TABLE.'.translatable_uuid')
            ->join(Capability::TABLE, fn ($join) => $join->on(DB::raw("
                CASE
                    WHEN inside_permissions.action = 'create' THEN 'assign'
                    ELSE inside_permissions.action
                END
            "), '=', Capability::TABLE.'.name'))
            ->leftjoin(CategorizableContentPrivilege::TABLE, function ($join) {
                $join->on(CategorizableContentTranslationIndexes::TABLE.'.uuid_host', '=', CategorizableContentPrivilege::TABLE.'.uuid_host')
                    ->on(Capability::TABLE.'.id', '=', CategorizableContentPrivilege::TABLE.'.capability_id');
            })
            ->whereIn('action', ['read', 'create', 'update', 'delete'])
            ->whereIn('type', $contentTypes->toArray())
            ->whereIn('role_id', $roles->pluck('id'))
            ->whereNull('user_uuid')
            ->select([
                'inside_permissions.type',
                CategorizableContentTranslationIndexes::TABLE.'.uuid_host',
                'inside_permissions.role_id',
                Capability::TABLE.'.name as action',
                Capability::TABLE.'.id as capability_id',
                CategorizableContentPrivilege::TABLE.'.id as privilege_id',
            ])
            ->get()
            ->map(fn ($permission) => [
                'role_id' => $permission->role_id,
                'privilege_id' => $permission->privilege_id,
            ])
            ->filter()
            ->groupBy('role_id');

        $authenticated = $roles->firstWhere('name', Role::AUTHENTICATED);

        $heritage = $privileges->get($authenticated->id)->pluck('privilege_id')->filter()->unique();

        $authenticated->categorizableContentPrivileges()->sync($heritage);

        $privileges->except($authenticated->id)->each(function ($privileges, $role) use ($roles, $heritage) {
            $privileges = $privileges->pluck('privilege_id')->filter()->unique()->diff($heritage);

            Log::info("[Inside Permission] Assigning privileges [{$privileges->count()}] to role [{$role}]");

            $roles->firstWhere('id', $role)->categorizableContentPrivileges()->sync($privileges);
        });
    }
}
