<?php

namespace Inside\Permission\Console;

use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Inside\Console\Command;
use Inside\Permission\Facades\Permission;
use Inside\Permission\Services\PermissionService;

class PermissionsV2RequirementCommand extends Command
{
    /**
     * @var string
     */
    protected $name = 'permissions:V2:requirement';

    /**
     * @var string
     */
    protected $description = 'Check if database is ready for v2';

    /**
     * @throws \Exception
     */
    public function handle(): int
    {
        $collations = collect();

        // Information of the database
        $defaultCharsetDatabase = DB::selectOne('SELECT DEFAULT_CHARACTER_SET_NAME FROM information_schema.SCHEMATA WHERE SCHEMA_NAME = DATABASE()');
        $defaultCollationDatabase = DB::selectOne('SELECT DEFAULT_COLLATION_NAME FROM information_schema.SCHEMATA WHERE SCHEMA_NAME = DATABASE()');
        $collations->push($defaultCollationDatabase->DEFAULT_COLLATION_NAME);

        // Information of the tables
        $tables = DB::select("SELECT TABLE_NAME, TABLE_COLLATION FROM information_schema.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_TYPE = 'BASE TABLE'");
        $columns = DB::select('SELECT TABLE_NAME, COLUMN_NAME, CHARACTER_SET_NAME, COLLATION_NAME FROM information_schema.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND COLLATION_NAME IS NOT NULL;');

        $collations->push(collect($tables)->pluck('TABLE_COLLATION')->unique()->all());
        $collations->push(collect($columns)->pluck('COLLATION_NAME')->unique()->all());

        // Information from .env
        $connection = config('database.default');
        $envCharset = config("database.connections.$connection.charset");
        $envCollation = config("database.connections.$connection.collation");
        $collations->push($envCollation);

        $this->info('Charset information:');
        $this->comment('Default charset: '.$defaultCharsetDatabase->DEFAULT_CHARACTER_SET_NAME);
        $this->comment('Charset from .env: '.$envCharset);
        $isCharsetSame = $defaultCharsetDatabase->DEFAULT_CHARACTER_SET_NAME === $envCharset;
        $this->write('Is well charset the same? '.($isCharsetSame ? 'Yes' : 'No'));
        $this->writeResult($isCharsetSame);

        $this->info("\nCollation information:");
        $this->comment('Default collation: '.$defaultCollationDatabase->DEFAULT_COLLATION_NAME);
        $this->comment('Collation from .env: '.$envCollation);
        $isCollationSame = $defaultCollationDatabase->DEFAULT_COLLATION_NAME === $envCollation;
        $this->write('Is well collation the same? '.($isCollationSame ? 'Yes' : 'No'));
        $this->writeResult($isCollationSame);

        $this->info("\nTables information:");
        collect($tables)->groupBy('TABLE_COLLATION')->each(function ($group, $collation) {
            $this->comment("Collation of {$group->count()} tables: $collation");
        });
        $isTableCollationSame = collect($tables)->every(function ($table) use ($defaultCollationDatabase) {
            return $table->TABLE_COLLATION === $defaultCollationDatabase->DEFAULT_COLLATION_NAME;
        });
        $this->write('Is well collation the same? '.($isTableCollationSame ? 'Yes' : 'No'));
        $this->writeResult($isTableCollationSame);

        $this->info("\nColumns information:");
        collect($columns)->groupBy('COLLATION_NAME')->each(function ($group, $collation) {
            $this->comment("Collation of {$group->count()} columns: $collation");
        });
        $isColumnCollationSame = collect($columns)->every(function ($column) use ($defaultCollationDatabase) {
            return $isCollationSame = in_array($column->COLLATION_NAME, [$defaultCollationDatabase->DEFAULT_COLLATION_NAME, 'ascii_general_ci']);
        });
        $this->write('Is well collation the same? '.($isColumnCollationSame ? 'Yes' : 'No'));
        $this->writeResult($isColumnCollationSame);

        if ($isCharsetSame && $isCollationSame && $isTableCollationSame && $isColumnCollationSame) {
            $this->info("\nDatabase is ready for v2 🔥");

            return self::SUCCESS;
        }
        $this->warning("\nDatabase is not ready for v2 😱");

        $this->info("\nFirst, make sure to change the charset and the collation of the database in your .env file as:");
        $this->comment("DB_CHARSET=$defaultCharsetDatabase->DEFAULT_CHARACTER_SET_NAME");
        $this->comment("DB_COLLATION=$defaultCollationDatabase->DEFAULT_COLLATION_NAME");

        // FOR MYSQL
        $this->title('FOR MYSQL');
        $this->comment('source .env');
        $this->info("\nCheck if the current user has the permission to change the charset and collation of the database.");
        $this->comment('mysql -u"$DB_USERNAME" -h "$DB_HOST" -p"$DB_PASSWORD" "$DB_DATABASE" -e "SHOW GRANTS FOR CURRENT_USER;"');
        $this->info("\nMake a dump of your database before doing anything for example:");
        $this->comment('mysqldump --add-drop-table --routines --events --triggers --set-charset -h"$DB_HOST" -u"$DB_USERNAME" -p"$DB_PASSWORD" "$DB_DATABASE" > "$DB_DATABASE"_$(date +%Y%m%d).sql');
        $this->info("\nYou can use the following commands to fix collations:");
        $collations->flatten()->unique()->diff(['ascii_general_ci', $defaultCollationDatabase->DEFAULT_COLLATION_NAME])->each(function ($collation) use ($defaultCollationDatabase) {
            match ($collation) {
                'ascii_bin' => $this->comment("sed -i 's/$collation/ascii_general_ci/g'".' "$DB_DATABASE"_$(date +%Y%m%d).sql'),
                default => $this->comment("sed -i 's/$collation/$defaultCollationDatabase->DEFAULT_COLLATION_NAME/g'".' "$DB_DATABASE"_$(date +%Y%m%d).sql'),
            };
        });
        $this->info("\nIt's now time to import the dump:");
        $this->comment('mysql -u"$DB_USERNAME" -h "$DB_HOST" -p"$DB_PASSWORD" "$DB_DATABASE" < "$DB_DATABASE"_$(date +%Y%m%d).sql');

        // FOR MARIADB
        $this->title('FOR MARIADB');
        $this->comment('source .env');
        $this->info("\nCheck if the current user has the permission to change the charset and collation of the database.");
        $this->comment('mariadb -u"$DB_USERNAME" -h "$DB_HOST" -p"$DB_PASSWORD" "$DB_DATABASE" -e "SHOW GRANTS FOR CURRENT_USER;"');
        $this->info("\nMake a dump of your database before doing anything for example:");
        $this->comment('mariadb-dump --add-drop-table --routines --events --triggers --set-charset -h"$DB_HOST" -u"$DB_USERNAME" -p"$DB_PASSWORD" "$DB_DATABASE" > "$DB_DATABASE"_$(date +%Y%m%d).sql');
        $this->info("\nYou can use the following commands to fix collations:");
        $collations->flatten()->unique()->diff(['ascii_general_ci', $defaultCollationDatabase->DEFAULT_COLLATION_NAME])->each(function ($collation) use ($defaultCollationDatabase) {
            match ($collation) {
                'ascii_bin' => $this->comment("sed -i 's/$collation/ascii_general_ci/g'".' "$DB_DATABASE"_$(date +%Y%m%d).sql'),
                default => $this->comment("sed -i 's/$collation/$defaultCollationDatabase->DEFAULT_COLLATION_NAME/g'".' "$DB_DATABASE"_$(date +%Y%m%d).sql'),
            };
        });
        $this->info("\nIt's now time to import the dump:");
        $this->comment('mariadb -u"$DB_USERNAME" -h "$DB_HOST" -p"$DB_PASSWORD" "$DB_DATABASE" < "$DB_DATABASE"_$(date +%Y%m%d).sql');

        // FOR MYSQL WITH DOCKER
        $this->title('FOR MYSQL WITH DOCKER');
        $this->comment('source .env');
        $this->info("\nCheck if the current user has the permission to change the charset and collation of the database.");
        $this->comment('docker compose exec -it database mysql -u"$DB_USERNAME" -h "$DB_HOST" -p"$DB_PASSWORD" "$DB_DATABASE" -e "SHOW GRANTS FOR CURRENT_USER;"');
        $this->info("\nMake a dump of your database before doing anything for example:");
        $this->comment('docker compose exec -T database mysqldump --add-drop-table --routines --events --triggers --set-charset -h"$DB_HOST" -u"$DB_USERNAME" -p"$DB_PASSWORD" "$DB_DATABASE" > "$DB_DATABASE"_$(date +%Y%m%d).sql');
        $this->info("\nYou can use the following commands to fix collations:");
        $collations->flatten()->unique()->diff(['ascii_general_ci', $defaultCollationDatabase->DEFAULT_COLLATION_NAME])->each(function ($collation) use ($defaultCollationDatabase) {
            match ($collation) {
                'ascii_bin' => $this->comment("sed -i 's/$collation/ascii_general_ci/g'".' "$DB_DATABASE"_$(date +%Y%m%d).sql'),
                default => $this->comment("sed -i 's/$collation/$defaultCollationDatabase->DEFAULT_COLLATION_NAME/g'".' "$DB_DATABASE"_$(date +%Y%m%d).sql'),
            };
        });
        $this->info("\nIt's now time to import the dump:");
        $this->comment('docker compose exec -T database mysql -u"$DB_USERNAME" -h "$DB_HOST" -p"$DB_PASSWORD" "$DB_DATABASE" < "$DB_DATABASE"_$(date +%Y%m%d).sql');

        // FOR MARIADB WITH DOCKER
        $this->title('FOR MARIADB WITH DOCKER');
        $this->comment('source .env');
        $this->info("\nCheck if the current user has the permission to change the charset and collation of the database.");
        $this->comment('docker compose exec -it database mariadb -u"$DB_USERNAME" -h "$DB_HOST" -p"$DB_PASSWORD" "$DB_DATABASE" -e "SHOW GRANTS FOR CURRENT_USER;"');
        $this->info("\nMake a dump of your database before doing anything for example:");
        $this->comment('docker compose exec -T database mariadb-dump --add-drop-table --routines --events --triggers --set-charset -h"$DB_HOST" -u"$DB_USERNAME" -p"$DB_PASSWORD" "$DB_DATABASE" > "$DB_DATABASE"_$(date +%Y%m%d).sql');
        $this->info("\nYou can use the following commands to fix collations:");
        $collations->flatten()->unique()->diff(['ascii_general_ci', $defaultCollationDatabase->DEFAULT_COLLATION_NAME])->each(function ($collation) use ($defaultCollationDatabase) {
            match ($collation) {
                'ascii_bin' => $this->comment("sed -i 's/$collation/ascii_general_ci/g'".' "$DB_DATABASE"_$(date +%Y%m%d).sql'),
                default => $this->comment("sed -i 's/$collation/$defaultCollationDatabase->DEFAULT_COLLATION_NAME/g'".' "$DB_DATABASE"_$(date +%Y%m%d).sql'),
            };
        });
        $this->info("\nIt's now time to import the dump:");
        $this->comment('docker compose exec -T database mariadb -u"$DB_USERNAME" -h "$DB_HOST" -p"$DB_PASSWORD" "$DB_DATABASE" < "$DB_DATABASE"_$(date +%Y%m%d).sql');

        return self::FAILURE;
    }

    private function title(string $title): void
    {
        $title = '# '.$title.' #';
        $this->writeln("\n<fg=blue;options=bold>".str_repeat('#', strlen($title)).'</>');
        $this->writeln("<fg=blue;options=bold>$title</>");
        $this->writeln('<fg=blue;options=bold>'.str_repeat('#', strlen($title))."</>\n");
    }
}
