<?php

namespace Inside\Permission\Console;

use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Inside\Console\Command;
use Inside\Content\Facades\Schema;
use Inside\Permission\Models\PermissionSchema;
use Inside\Permission\Models\Role;
use Inside\Support\Str;

class CleanupCommand extends Command
{
    protected $name = 'inside:permission:cleanup';

    protected $signature = 'inside:permission:cleanup {--with-non-permissible : Tente de nettoyer les permissions de contenu qui ne sont plus permissible}';

    protected $description = 'Nettoie les incohérences de permissions';

    public function handle(): void
    {
        $this->info('Début de suppression des incohérences de permissions');

        $toDelete = collect();
        $uniques = collect();

        /** @var Collection<PermissionSchema> $schemas */
        $schemas = PermissionSchema::select('id', 'invert', 'children', 'action', 'authorizable_type', 'authorizable_uuid')->get();
        foreach ($schemas as $schema) {
            /** @var PermissionSchema $schema */
            if ($toDelete->contains($schema->id)) {
                continue; // Already done
            }
            // schema_unique
            //invert    4    A    Non
            //children    4    A    Non
            //action    16    A    Oui
            //authorizable_type    264    A    Oui
            //authorizable_uuid
            $uniques[] = $schema->id;
            $duplicates = $schemas
                ->where('invert', $schema->invert)
                ->where('children', $schema->children)
                ->where('action', $schema->action)
                ->where('authorizable_type', $schema->authorizable_type)
                ->where('authorizable_uuid', $schema->authorizable_uuid)
                ->whereNotIn('id', $uniques)
                ->sortByDesc('invert');

            if ($duplicates->isEmpty()) {
                continue; // No duplicate, woot !
            }

            $roles = $schema->roles;
            foreach ($duplicates as $duplicate) {
                $roles = $roles->merge($duplicate->roles);
            }
            // Attach roles
            DB::table('inside_roles_permissions_schema')->whereIn('permission_schema_id', $duplicates->pluck('id'))
                ->delete();
            $schema->roles()->sync($roles);
            $toDelete = $toDelete->merge($duplicates->pluck('id'));
        }

        // Remove uniques from toDelete
        $this->line(count($toDelete).' Schéma(s) de permission à effacer');
        foreach ($toDelete->chunk(1000) as $ids) {
            DB::table('inside_permissions_schema')->whereIn('id', $ids)->delete();
        }
        $this->line('Schémas OK');

        if ($this->option('with-non-permissible')) {
            $nonPermissibleContent = Schema::getContentTypes(function ($options) {
                return ! isset($options['options']) || ! isset($options['options']['permissible'])
                    || $options['options']['permissible'] != true;
            });
            $result = [];
            foreach ($nonPermissibleContent as $type) {
                // Check permission schema
                $result[] = [
                    'type'        => $type,
                    'schemas'     => DB::table('inside_permissions_schema')->where(
                        'authorizable_type',
                        type_to_class($type)
                    )->count(),
                    'permissions' => DB::table('inside_permissions')->where('type', type_to_class($type))->count(),
                ];
            }
            $this->table(['type', 'schemas', 'permissions'], $result);
            if ($this->confirm(
                'Vous êtes sur le point de supprimer les permissions correspondantes, êtes vous sur de vouloir continuer ?'
            )
            ) {
                foreach ($result as $data) {
                    if ($data['permissions'] === 0 && $data['schemas'] === 0) {
                        continue;
                    }
                    $this->write('Suppression des permissions du type <fg=cyan>'.$data['type'].'</fg=cyan>');
                    $deleted = DB::table('inside_permissions')->where('type', type_to_class($data['type']))->delete();
                    $this->writeResult($deleted == $data['permissions']);
                    $this->warn($deleted.' permissions supprimées');
                    $this->write(
                        'Suppression des schemas de permissions du type <fg=cyan>'.$data['type'].'</fg=cyan>'
                    );
                    $deleted = DB::table('inside_permissions_schema')->where(
                        'authorizable_type',
                        type_to_class($data['type'])
                    )->delete();
                    $this->writeResult($deleted == $data['schemas']);
                    $this->warn($deleted.' schemas supprimés');
                }
            }
        }

        // Remove categorizable schema on non categorizable permission
        $this->line('Suppression des schémas invalide provenant d\'un ancien règlage');
        $nonCategorizablePermissibleTypes = Schema::getContentTypes(function ($model) {
            return isset($model['options']['permissible']) && $model['options']['permissible']
                && (! isset($model['options']['categorizable']) || ! $model['options']['categorizable'])
                && ! Str::endsWith($model['class'], 'Menus');
        });
        Role::each(function ($role) use ($nonCategorizablePermissibleTypes) {
            $this->warning('Schema du role ['.\Inside\Permission\Facades\Role::getHumanName($role->name).']');

            foreach ($nonCategorizablePermissibleTypes as $type) {
                $this->info('Recherche de schéma erroné de type ['.$type.']');
                $query = DB::table('inside_permissions_schema')->join(
                    'inside_roles_permissions_schema',
                    'inside_permissions_schema.id',
                    'inside_roles_permissions_schema.permission_schema_id'
                )->where('inside_roles_permissions_schema.role_id', $role->id)->where(
                    'inside_roles_permissions_schema.is_content_specific',
                    0
                )->whereNotNull('inside_permissions_schema.authorizable_uuid')->where(
                    'inside_permissions_schema.authorizable_type',
                    type_to_class($type)
                )->whereNull('custom');
                if ($query->count() > 0) {
                    $this->error(
                        'Nous avons détecté ('.$query->count().') schema(s) spécifique au type ['.$type.'] '
                    );
                    $query->delete();
                }
            }
        });
    }
}
